package ru.yandex.direct.validation.constraint;

import java.net.IDN;
import java.util.List;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import com.google.common.base.CharMatcher;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.validator.routines.EmailValidator;

import ru.yandex.direct.utils.UrlUtils;
import ru.yandex.direct.validation.Predicates;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.defect.CollectionDefects;
import ru.yandex.direct.validation.defect.NumberDefects;
import ru.yandex.direct.validation.defect.StringDefects;
import ru.yandex.direct.validation.result.Defect;

import static ru.yandex.direct.utils.TextConstants.ALL;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.defect.CommonDefects.invalidFormat;
import static ru.yandex.direct.validation.defect.CommonDefects.invalidValue;

public class StringConstraints {
    public static final String NOT_CONTROL_CHARS = "\\P{Cntrl}*";
    public static final String PROTOCOL_PATTERN = "(?:https?://)";
    private static final Pattern PROTOCOL_PATTERN_COMPILED = Pattern.compile(PROTOCOL_PATTERN);
    private static final String DOMAIN_LABEL_PATTERN = "(?:[\\p{Alpha}0-9][\\p{Alpha}0-9\\-_]{0,62}\\.)";
    private static final String TOP_LEVEL_DOMAIN_PATTERN = "\\p{Alpha}[\\p{Alpha}0-9\\-]{1,14}";
    private static final String IP_V4_NUMBER_PATTERN = "(?:[0-9]|[1-9][0-9]|1[0-9][0-9]|2[0-4][0-9]|25[0-5])";
    private static final String IP_V4_PATTERN =
            "(?:" + IP_V4_NUMBER_PATTERN + "\\.){3}" + IP_V4_NUMBER_PATTERN;
    public static final String DOMAIN_PATTERN =
            "(" + DOMAIN_LABEL_PATTERN + "+" + TOP_LEVEL_DOMAIN_PATTERN + ")";
    public static final String DOMAIN_OR_IP_PATTERN =
            "((?:" + DOMAIN_LABEL_PATTERN + "+" + TOP_LEVEL_DOMAIN_PATTERN + ")|" + IP_V4_PATTERN + ")";
    public static final String PORT_PATTERN = "(?::[0-9]{1,5})?";
    private static final Pattern HREF_PATTERN = Pattern.compile("^"
            + PROTOCOL_PATTERN  // protocol
            + DOMAIN_PATTERN // domain, group #1
            + PORT_PATTERN // port
            + "(?:[/?#][\\[\\]\\-\\w/=&%#?():;.,~+{}\\^\\@\\$\\*!\\|\\']*)?" // path and query parts
            + "\\z", Pattern.UNICODE_CHARACTER_CLASS);

    /**
     * Почти то же самое, что и {@link #HREF_PATTERN}, но допускает в домене IPv4 адреса, что является допустимым
     * в соответствии с <a href="https://datatracker.ietf.org/doc/html/rfc3986#section-3.2.2">RFC</a>.
     * <p>
     * Почему мы не используем один паттерн для всего: в некоторых ситуациях IP адрес в качестве домена
     * может быть недопустим.
     * <p>
     * Важно: в данный момент этот паттерн не поддерживает IPv6 адреса, хотя RFC это допускает.
     */
    private static final Pattern URL_PATTERN = Pattern.compile("^"
            + PROTOCOL_PATTERN  // protocol
            + DOMAIN_OR_IP_PATTERN // domain, group #1
            + PORT_PATTERN // port
            + "(?:[/?#][\\[\\]\\-\\w/=&%#?():;.,~+{}\\^\\@\\$\\*!\\|\\']*)?" // path and query parts
            + "\\z", Pattern.UNICODE_CHARACTER_CLASS);
    private static final AdmissibleCharsConstraint ADMISSIBLE_CHARS_CONSTRAINT = new AdmissibleCharsConstraint(ALL);
    private static final CharMatcher DELIM_MATCHER = CharMatcher.anyOf("/?");
    private static final CharMatcher PARAMS_MARKER_MATCHER = CharMatcher.anyOf("{}");
    private static final CharMatcher TEMPLATE_MARKER_MATCHER = CharMatcher.is('#').or(PARAMS_MARKER_MATCHER);

    public static Constraint<String, Defect> isPositiveWholeNumber() {
        return fromPredicate(Predicates.isPositiveWholeNumber(), NumberDefects.isWholeNumber());
    }

    public static Constraint<String, Defect> notEmpty2() {
        return fromPredicate(StringUtils::isNotEmpty, StringDefects.notEmptyString());
    }

    /**
     * @deprecated Используйте {@link #notBlank()}, так как есть необходимость в отдельном констрейнте, который бы
     * проверял только пустую строку - иногда у пустой строки и строки из пробелов разные ошибки
     */
    @Deprecated
    public static Constraint<String, Defect> notEmpty() {
        return fromPredicate(StringUtils::isNotBlank, StringDefects.notEmptyString());
    }

    public static Constraint<String, Defect> notBlank() {
        return fromPredicate(StringUtils::isNotBlank, StringDefects.notEmptyString());
    }

    public static Constraint<List<String>, Defect> eachNotBlank() {
        return fromPredicate(
                list -> list.stream().allMatch(StringUtils::isNotBlank), invalidValue());
    }

    public static Constraint<Set<String>, Defect> eachInSetNotBlank() {
        return Constraint.fromPredicate(
                list -> list.stream().allMatch(StringUtils::isNotBlank), invalidValue());
    }

    public static Constraint<String, Defect> minStringLength(int min) {
        return fromPredicate(Predicates.minLength(min), CollectionDefects.minStringLength(min));
    }

    public static Constraint<String, Defect> maxStringLength(int max) {
        return fromPredicate(Predicates.maxLength(max), CollectionDefects.maxStringLength(max));
    }

    public static Constraint<String, Defect> admissibleChars() {
        return ADMISSIBLE_CHARS_CONSTRAINT;
    }

    public static Constraint<String, Defect> hasNoForbiddenChars(List<String> forbiddenChars) {
        return fromPredicate(Predicates.notContains(forbiddenChars),
                StringDefects.mustNotContainsForbiddenSymbols(forbiddenChars));
    }

    public static Constraint<String, Defect> notControlChars() {
        return matchPattern(NOT_CONTROL_CHARS);
    }

    public static Constraint<String, Defect> notControlChars_AdmissibleChars() {
        Pattern compiledPattern = Pattern.compile(NOT_CONTROL_CHARS);
        return fromPredicate(v -> compiledPattern.matcher(v).matches(), StringDefects.admissibleChars());
    }

    /**
     * Проверяет что строка содержит только utf8mb3, которые можно записать в базу
     */
    public static Constraint<String, Defect> onlyUtf8Mb3Symbols() {
        return fromPredicate(name -> name.chars().noneMatch(c -> Character.isSurrogate((char) c)),
                StringDefects.admissibleChars());
    }

    /**
     * Проверить на соответствие регулярному выражению
     *
     * @param pattern регулярное выражение
     */
    public static Constraint<String, Defect> matchPattern(String pattern) {
        Pattern compiledPattern = Pattern.compile(pattern);
        return fromPredicate(v -> compiledPattern.matcher(v).matches(), invalidValue());
    }

    /**
     * Проверить на соответствие регулярному выражению
     *
     * @param pattern регулярное выражение
     */
    public static Constraint<String, Defect> matchPattern(Pattern pattern) {
        return fromPredicate(v -> pattern.matcher(v).matches(), invalidValue());
    }

    /**
     * Проверить корректность email с помощью Apache EmailValidator
     */
    public static Constraint<String, Defect> validEmail() {
        return validEmail(false);
    }

    public static Constraint<String, Defect> validEmail(boolean checkOnlyAtSymbol) {
        return validEmail(checkOnlyAtSymbol, invalidValue());
    }

    public static Constraint<String, Defect> validEmailInvalidFormat() {
        return validEmailInvalidFormat(false);
    }

    public static Constraint<String, Defect> validEmailInvalidFormat(boolean checkOnlyAtSymbol) {
        return validEmail(checkOnlyAtSymbol, invalidFormat());
    }

    private static Constraint<String, Defect> validEmail(boolean checkOnlyAtSymbol, Defect defect) {
        if (checkOnlyAtSymbol) {
            return fromPredicate(email -> email.contains("@"), defect);
        }
        return fromPredicate(
                v -> EmailValidator.getInstance().isValid(v),
                defect);
    }

    public static Constraint<String, Defect> contains(Pattern pattern) {
        return fromPredicate(pattern.asPredicate(), invalidValue());
    }

    public static Constraint<String, Defect> notContains(Pattern pattern) {
        return fromPredicate(v -> !pattern.matcher(v).find(), invalidValue());
    }

    /**
     * Провека валидности ссылки (полная проверка, включая Unicode URL'ы)
     */
    public static Constraint<String, Defect> validHref() {
        return fromPredicate(StringConstraints::isValidHref, invalidValue());
    }

    public static boolean isValidHref(String href) {
        // проверки на null и пустое значение делаются отдельно
        if (StringUtils.isBlank(href)) {
            return true;
        }

        String hrefEncoded = UrlUtils.urlPathToAsciiIfCan(href);
        Matcher matcher = HREF_PATTERN.matcher(hrefEncoded);
        if (!matcher.matches()) {
            return false;
        }

        if (checkForDomainTemplates(hrefEncoded)) {
            return false;
        }

        String domain = matcher.group(1);
        if (CharMatcher.ascii().matchesAllOf(domain)) {
            return true;
        }

        try {
            IDN.toASCII(domain);
            return true;
        } catch (IllegalArgumentException e) {
            return false;
        }
    }

    public static Constraint<String, Defect> validUrl() {
        return fromPredicate(StringConstraints::isValidUrl, invalidValue());
    }

    public static boolean isValidUrl(String url) {
        // проверки на null и пустое значение делаются отдельно
        if (StringUtils.isBlank(url)) {
            return true;
        }

        String urlEncoded = UrlUtils.urlPathToAsciiIfCan(url);
        var matcher = URL_PATTERN.matcher(urlEncoded);
        return matcher.matches();
    }

    /**
     * Проверка валидности трекинговых параметров, приклеивающихся к ссылкам на объявлениях.
     * Реализована через небольшой хак: трекинговые параметры валидны тогда,
     * когда ссылка, получающаяся после их приклеивания к заведомо верному url, валидная
     * */
    public static Constraint<String, Defect> validTrackingParams() {
        return fromPredicate(StringConstraints::isValidTrackingParams, invalidValue());
    }

    public static boolean isValidTrackingParams(String trackingParams) {
        return isValidHref("https://yandex.ru/?" + trackingParams);
    }

    /**
     * Проверить, нет ли подстановочных параметров в доменной части у ссылки.
     * Параметры обозначаются либо парой {}, либо парой ##. Доменная часть должна заканчиваться '/' или '?', иногда '#'.
     * Также замечает единичные '#', '{', '}' до '/' или '?'.
     *
     * @param href — проверяемая ссылка
     * @return правда, если подстановочные параметры есть в доменной части ссылки
     */
    private static boolean checkForDomainTemplates(String href) {
        String hrefCopy = PROTOCOL_PATTERN_COMPILED.matcher(href).replaceFirst("");
        int index = DELIM_MATCHER.indexIn(hrefCopy);
        if (index >= 0) {
            hrefCopy = hrefCopy.substring(0, index);
            return TEMPLATE_MARKER_MATCHER.matchesAnyOf(hrefCopy);
        }
        return StringUtils.countMatches(hrefCopy, '#') > 1 || PARAMS_MARKER_MATCHER.matchesAnyOf(hrefCopy);
    }

    public static Constraint<String, Defect> startWith(String prefix) {
        return Constraint.fromPredicate(t -> t.startsWith(prefix), invalidValue());
    }
}
