package ru.yandex.direct.validation.constraint;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.function.Supplier;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.validation.builder.ListConstraint;

import static java.util.stream.Collectors.counting;
import static java.util.stream.Collectors.groupingBy;

/***
 * Проверяет, уникален ли элемент списка. Более абстрактная версия чем SimpleUniqueItemsConstraint.
 *
 * У проверяемых объектов должны быть валидные методы equals/hashCode.
 *
 * @param <I> Тип элемента в списке
 * @param <R> Тип данных, которые сравнивают
 * @param <D> Тип дефекта, который возвращается, если элемент совпал
 */
@ParametersAreNonnullByDefault
public class UniqueItemsConstraint<I, R, D> implements ListConstraint<I, D> {
    private final Function<I, R> getter;
    private final Supplier<D> defectSupplier;

    public UniqueItemsConstraint(Function<I, R> getter, Supplier<D> defectSupplier) {
        this.getter = t -> t != null ? getter.apply(t) : null;
        this.defectSupplier = defectSupplier;
    }

    /**
     * Применить проверку к списку.
     *
     * @param list Список для проверки.
     * @return Хэш-таблицу, где ключи - номера элементов массива с дефектами, а значения - сами дефекты
     */
    @Override
    @Nonnull
    public Map<Integer, D> apply(List<I> list) {
        Map<Integer, D> defectMap = new HashMap<>();
        Map<R, Long> itemCounts = list.stream().collect(groupingBy(getter, counting()));

        for (int idx = 0; idx < list.size(); idx++) {
            R item = getter.apply(list.get(idx));
            if (item == null || itemCounts.get(item) <= 1) {
                continue;
            }
            defectMap.put(idx, defectSupplier.get());
        }

        return defectMap;
    }
}
