package ru.yandex.direct.validation.presentation;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;

import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.DefectId;
import ru.yandex.direct.validation.result.DefectInfo;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.collect.Sets.intersection;

/**
 * Дефолтная реализация интерфейса {@link DefectPresentationRegistry}.
 * Реестр, содержащий мапу id ядровых дефектов {@link DefectId} на
 * провайдеры представлений для этих дефектов - {@link DefectPresentationProvider}.
 * <p>
 * Его можно собрать как из отдельных провайдеров, так и из других реестров.
 * Последнее может быть полезно, когда хочется сгруппировать регистрацию дефектов,
 * в этом случае отдельно собираются несколько логически разделенных реестров,
 * а затем из них собирается один общий.
 *
 * @param <P> тип представления дефекта.
 */
@ParametersAreNonnullByDefault
public class DefaultDefectPresentationRegistry<P> implements DefectPresentationRegistry<P> {

    /**
     * Несмотря на то, что в мапе типы получаемых на вход дефектов
     * не определены, методы билдера гарантируют, что для каждого
     * {@code DefectId<T>} зарегистрирован провайдер совместимого типа:
     * {@code DefectPresentationProvider<T, P>}.
     */
    private final ImmutableMap<DefectId<?>, DefectPresentationProvider<?, P>> presentationProviderMap;

    private DefaultDefectPresentationRegistry(
            Map<DefectId<?>, DefectPresentationProvider<?, P>> presentationProviderMap) {
        this.presentationProviderMap = ImmutableMap.copyOf(presentationProviderMap);
    }

    @Nullable
    @SuppressWarnings("unchecked")
    public P getPresentation(DefectInfo<? extends Defect> defectInfo) {
        //noinspection unchecked
        DefectPresentationProvider presentationProvider =
                presentationProviderMap.get(defectInfo.getDefect().defectId());
        return presentationProvider == null ? null :
                (P) presentationProvider.getPresentation(defectInfo);
    }

    @Override
    public Set<DefectId<?>> getRegisteredDefectIds() {
        return presentationProviderMap.keySet();
    }

    public static UntypedBuilder builder() {
        return new UntypedBuilder();
    }

    public static class UntypedBuilder {

        protected UntypedBuilder() {
        }

        public <T, P> Builder<P> register(DefectId<T> defectId, DefectPresentationProvider<T, P> provider) {
            return new Builder<P>().register(defectId, provider);
        }

        public <P> Builder<P> register(DefaultDefectPresentationRegistry<P> anotherRegistry) {
            return new Builder<P>().register(anotherRegistry);
        }
    }

    public static class Builder<P> {

        private Map<DefectId<?>, DefectPresentationProvider<?, P>> registrationMap = new HashMap<>();

        protected Builder() {
        }

        public <T> Builder<P> register(DefectId<T> defectId, DefectPresentationProvider<T, P> provider) {
            checkDuplication(defectId);
            registrationMap.put(defectId, provider);
            return this;
        }

        public Builder<P> register(DefaultDefectPresentationRegistry<P> anotherRegistry) {
            checkDuplication(anotherRegistry.presentationProviderMap.keySet());
            registrationMap.putAll(anotherRegistry.presentationProviderMap);
            return this;
        }

        public DefaultDefectPresentationRegistry<P> build() {
            return new DefaultDefectPresentationRegistry<>(registrationMap);
        }

        private void checkDuplication(DefectId<?> newDefectId) {
            checkArgument(!registrationMap.containsKey(newDefectId),
                    "attempt to register more than one defect presentation provider for defect id: %s",
                    newDefectId);
        }

        private void checkDuplication(Set<DefectId<?>> newDefectIds) {
            Set<DefectId<?>> duplicatedKeys = intersection(registrationMap.keySet(), newDefectIds);
            checkArgument(duplicatedKeys.isEmpty(),
                    "attempt to register more than one defect presentation provider for defect ids: %s",
                    duplicatedKeys);
        }
    }
}
