package ru.yandex.direct.validation.presentation;

import java.util.Set;
import java.util.function.Function;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Sets;

import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.DefectId;
import ru.yandex.direct.validation.result.DefectInfo;

import static com.google.common.base.Preconditions.checkState;

@ParametersAreNonnullByDefault
public class DefectPresentationRegistryFactory {

    /**
     * Оборачивает переданный реестр в декоратор, который выкидывает исключение,
     * когда представление не найдено.
     *
     * @param registry исходный реестр
     * @param <X>      тип представления
     */
    public static <X> DefectPresentationRegistry<X> strictRegistry(DefectPresentationRegistry<X> registry) {
        return new DefectPresentationRegistry<X>() {

            @Override
            public X getPresentation(DefectInfo<? extends Defect> defectInfo) {
                X presentation = registry.getPresentation(defectInfo);
                checkState(presentation != null, "no defect presentation provider was found for %s",
                        defectInfo.getDefect().defectId());
                return presentation;
            }

            @Override
            public Set<DefectId<?>> getRegisteredDefectIds() {
                return registry.getRegisteredDefectIds();
            }
        };
    }

    /**
     * Оборачивает переданный реестр в декоратор, который преобразовывает
     * представление первого декоратора в другое представление.
     * <p>
     * Может быть полезен, когда для конвертации в конечное представление требуется
     * использовать ряд сервисов, и при этом не хочется прокидывать их в каждый
     * {@link DefectPresentationProvider} при инициализации {@link DefaultDefectPresentationRegistry}.
     * <p>
     * Например, исходный реестр преобразует {@link DefectInfo} в сущность,
     * содержащую {@link ru.yandex.direct.i18n.Translatable}, а декоратор
     * берет на себя конвертацию в строку.
     *
     * @param registry исходный реестр.
     * @param <X>      тип представления.
     */
    public static <X, Y> DefectPresentationRegistry<Y> conversionRegistry(DefectPresentationRegistry<X> registry,
                                                                          Function<X, Y> conversionFunction) {
        return new DefectPresentationRegistry<Y>() {

            @Nullable
            @Override
            public Y getPresentation(DefectInfo<? extends Defect> defectInfo) {
                X presentation = registry.getPresentation(defectInfo);
                return presentation != null ? conversionFunction.apply(presentation) : null;
            }

            @Override
            public Set<DefectId<?>> getRegisteredDefectIds() {
                return registry.getRegisteredDefectIds();
            }
        };
    }

    /**
     * Создает реестр, который обращается к первому переданному реестру с фоллбэком на второй.
     *
     * @param registry первый реестр
     * @param fallback фоллбэк-реестр
     * @param <X>      тип презентации
     */
    public static <X> DefectPresentationRegistry<X> fallbackRegistry(DefectPresentationRegistry<X> registry,
                                                                     DefectPresentationRegistry<X> fallback) {
        return new DefectPresentationRegistry<X>() {

            @Nullable
            @Override
            public X getPresentation(DefectInfo<? extends Defect> defectInfo) {
                X presentation = registry.getPresentation(defectInfo);
                if (presentation == null) {
                    return fallback.getPresentation(defectInfo);
                }
                return presentation;
            }

            @Override
            public Set<DefectId<?>> getRegisteredDefectIds() {
                return Sets.union(registry.getRegisteredDefectIds(), fallback.getRegisteredDefectIds());
            }
        };
    }
}
