package ru.yandex.direct.validation.result;

import java.util.HashMap;
import java.util.Map;

import com.google.common.collect.ImmutableMap;

import static com.google.common.base.Preconditions.checkArgument;

public class DefaultPathNodeConverterProvider implements PathNodeConverterProvider {

    private final ImmutableMap<Class, PathNodeConverter> converterMap;
    private final PathNodeConverterProvider fallbackProvider;

    private DefaultPathNodeConverterProvider(
            ImmutableMap<Class, PathNodeConverter> converterMap,
            PathNodeConverterProvider fallbackProvider) {
        this.converterMap = converterMap;
        this.fallbackProvider = fallbackProvider;
    }

    @Override
    public PathNodeConverter getConverter(Class clazz) {
        PathNodeConverter converter = converterMap.get(clazz);
        if (converter == null && fallbackProvider != null) {
            return fallbackProvider.getConverter(clazz);
        }
        return converter;
    }

    public static Builder builder() {
        return new Builder();
    }

    public static class Builder {
        private Map<Class, PathNodeConverter> tmpMap = new HashMap<>();
        private PathNodeConverterProvider fallbackProvider;

        public Builder register(Class clazz, PathNodeConverter pathNodeConverter) {
            checkArgument(!tmpMap.containsKey(clazz),
                    "PathNodeConverter for class %s has been already registered", clazz);
            tmpMap.put(clazz, pathNodeConverter);
            return this;
        }

        public Builder fallbackTo(PathNodeConverterProvider fallbackProvider) {
            checkArgument(this.fallbackProvider == null,
                    "fallback PathNodeConverterProvider has been already set");
            this.fallbackProvider = fallbackProvider;
            return this;
        }

        public DefaultPathNodeConverterProvider build() {
            return new DefaultPathNodeConverterProvider(ImmutableMap.copyOf(tmpMap), fallbackProvider);
        }
    }
}
