package ru.yandex.direct.validation.result;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Stream;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;

import static com.google.common.base.Preconditions.checkArgument;
import static java.util.stream.Collectors.toList;

/**
 * Конвертирует {@link Path} в соответствии с заданными правилами подстановки.
 * <p>
 * Пример использования:
 * <pre>PathConverter pathConverter = PathConverter.builder()
 *     .add("priceContext", "ContextBid")
 *     .add("retargetings", "AudienceTargets")
 *     .build();
 *
 * Path corePath = pathFromString("retargetings[0].priceContext");
 * Path apiPath = pathConverter.convert(corePath);
 * apiPath.toString(); // AudienceTargets[0].ContextBid</pre>
 *
 * @see Path
 * @see PathNode
 * @see PathHelper
 */
@ParametersAreNonnullByDefault
public class MappingPathConverter implements PathConverter {
    private final String name;
    private final Map<String, String> pathDict;
    private final Set<String> ignoreIndex;

    private boolean ignoreIndexFlag = false;

    private MappingPathConverter(String name, Map<String, String> pathDict, Set<String> ignoreIndex) {
        this.name = name;
        this.pathDict = pathDict;
        this.ignoreIndex = ignoreIndex;
    }

    /**
     * @deprecated надо использовать конструктор с названиием {@link #builder(Class, String)}
     */
    @Deprecated
    public static Builder builder() {
        @SuppressWarnings("ThrowableNotThrown")
        StackTraceElement traceElement = new Exception().getStackTrace()[1];
        String converterName = traceElement.getClassName() +
                ":" + traceElement.getLineNumber() +
                "#" + traceElement.getMethodName();
        return builder(converterName);
    }

    /**
     * @param clazz класс, в котором создаётся {@link MappingPathConverter}.
     * @param name  имя конвертера
     * @return экземпляр {@link Builder}
     */
    public static Builder builder(Class<?> clazz, String name) {
        String converterName = clazz.getSimpleName() + "#" + name;
        return builder(converterName);
    }

    /**
     * @param name имя конвертера
     * @return экземпляр {@link Builder}
     */
    static Builder builder(String name) {
        return new Builder(name);
    }

    /**
     * Преобразует {@link Path} в соответствии с заданными правилами подстановки.
     *
     * @param source {@link Path} для преобразования
     * @return экземпляр {@link Path} с заменёнными узлами
     * @throws IllegalStateException если в словаре не нашлось подстановки
     */
    @Override
    @Nonnull
    public Path convert(Path source) throws IllegalStateException {
        List<PathNode> newNodes = new ArrayList<>();
        for (PathNode sourceNode : source.getNodes()) {
            if (sourceNode instanceof PathNode.Index) {
                if (!ignoreIndexFlag) {
                    newNodes.add(sourceNode);
                }
                ignoreIndexFlag = false;
            } else {
                newNodes.addAll(convertSimpleNode((PathNode.Field) sourceNode));
            }
        }
        return new Path(newNodes);
    }

    private List<PathNode> convertSimpleNode(PathNode.Field node) throws IllegalStateException {
        if (!pathDict.containsKey(node.getName())) {
            throw new IllegalStateException(String.format(
                    "Can't find matching node in converter dict. It may point to path converting error. "
                            + "Node: %s. Converter: %s",
                    node.getName(), name));
        } else {
            ignoreIndexFlag = ignoreIndex.contains(node.getName());
            String newNodePath = pathDict.get(node.getName());
            return Stream
                    .of(newNodePath.split("\\."))
                    .filter(s -> !s.isEmpty())
                    .map(PathNode.Field::new)
                    .collect(toList());
        }
    }

    /**
     * Конструктор для {@link MappingPathConverter}. Стандартный способ создания экземпляра builder'а &ndash;
     * через {@link MappingPathConverter#builder(String)}
     */
    public static class Builder {
        private final String name;
        private final ImmutableMap.Builder<String, String> builder;
        private final ImmutableSet.Builder<String> ignoreIndexBuilder;

        private Builder(String name) {
            this.name = name;
            builder = ImmutableMap.builder();
            ignoreIndexBuilder = ImmutableSet.builder();
        }

        public Builder add(Map<String, String> dict) {
            builder.putAll(dict);
            return this;
        }

        public Builder add(String key, String value) {
            checkArgument(!value.isEmpty(), "use ignore() instead of passing empty string");
            builder.put(key, value);
            return this;
        }

        /**
         * Игнорировать элемент пути.
         * <p>
         * Указаного элемента пути не будет в результате конвертации.
         *
         * @param key какой элемент пути удалить из результата
         */
        public Builder ignore(String key) {
            builder.put(key, "");
            return this;
        }

        /**
         * Игнорирует индекс, следующий за указанным элементом пути.
         * <p>
         * Индекса указанного элемента пути не будет в результате конвертации.
         *
         * @param key индекс какого элемент пути удалить из результата
         */
        public Builder ignoreIndex(String key) {
            ignoreIndexBuilder.add(key);
            return this;
        }

        public PathConverter build() {
            return new MappingPathConverter(name, builder.build(), ignoreIndexBuilder.build());
        }
    }
}
