package ru.yandex.direct.validation.result;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.google.common.collect.ImmutableMap;
import one.util.streamex.EntryStream;

import static com.google.common.base.Preconditions.checkArgument;
import static java.util.Collections.emptyList;
import static java.util.Collections.singletonList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.result.PathHelper.concat;
import static ru.yandex.direct.validation.result.PathHelper.path;

/**
 * Конвертирует {@link PathNode} в соответствии с заданными правилами подстановки.
 * <p>
 * Пример использования:
 * <pre>PathNodeConverter converter = MappingPathNodeConverter.builder("showConditions")
 *     .replace("priceContext", "ContextBid")
 *     .replace("retargetings", "AudienceTargets")
 *     .build();
 * </pre>
 *
 * @see PathNode
 * @see PathNodeConverterProvider
 */
public class MappingPathNodeConverter implements PathNodeConverter {

    private final String name;
    private final ImmutableMap<PathNode.Field, Path> singleItemDict;
    private final ImmutableMap<PathNode.Field, Path> listItemDict;

    MappingPathNodeConverter(String name, ImmutableMap<PathNode.Field, Path> pathDict,
                             ImmutableMap<PathNode.Field, Path> listItemDict) {
        this.name = name;
        this.singleItemDict = pathDict;
        this.listItemDict = listItemDict;
    }

    public static MappingPathNodeConverter.Builder builder(String name) {
        return new MappingPathNodeConverter.Builder(name);
    }

    @Override
    public Path convert(PathNode.Field fieldName) {
        if (!singleItemDict.containsKey(fieldName)) {
            return path(fieldName);
        } else {
            return singleItemDict.get(fieldName);
        }
    }

    @Override
    public Path convert(PathNode.Field fieldName, PathNode.Index sourceIndex) {
        if (!listItemDict.containsKey(fieldName)) {
            return path(sourceIndex);
        } else {
            return concat(sourceIndex, listItemDict.get(fieldName));
        }
    }

    public static class Builder {
        private final String name;
        private final Map<String, List<String>> singleItemDict = new HashMap<>();
        private final Map<String, List<String>> listItemDict = new HashMap<>();

        private Builder(String name) {
            this.name = name;
        }

        /**
         * Преобразование одного элемента исходного пути в один элемент результирующего пути.
         *
         * @param sourceFieldName исходное название
         * @param destFieldName   во что преобразуется
         */
        public MappingPathNodeConverter.Builder replace(String sourceFieldName, String destFieldName) {
            checkArgument(!singleItemDict.containsKey(sourceFieldName), "dictionary already contains %s",
                    sourceFieldName);
            singleItemDict.put(sourceFieldName, singletonList(destFieldName));
            return this;
        }

        /**
         * Преобразование одного элемента исходного пути в несколько элементов результирующего пути.
         * <p>
         * Используется, если нужно добавить в путь промежуточные элементы.
         * Например, исходный путь: {@code banner.creativeId},
         * а в результате хочется получить {@code banner.videoResources.creativeId}.
         *
         * @param sourceFieldName исходное название
         * @param destFieldNames  во что преобразуется
         */
        public MappingPathNodeConverter.Builder replace(String sourceFieldName, List<String> destFieldNames) {
            checkArgument(!singleItemDict.containsKey(sourceFieldName), "dictionary already contains %s",
                    sourceFieldName);
            singleItemDict.put(sourceFieldName, destFieldNames);
            return this;
        }

        /**
         * Удаление элемента исходного пути.
         * <p>
         * Например, исходный путь: {@code banner.sitelinkSet.sitelinks},
         * а в результате хочется получить {@code banner.sitelinks}.
         *
         * @param sourceFieldName название ноды в исходном пути, которую необходимо удалить
         */
        public MappingPathNodeConverter.Builder skip(String sourceFieldName) {
            checkArgument(!singleItemDict.containsKey(sourceFieldName), "dictionary already contains %s",
                    sourceFieldName);
            singleItemDict.put(sourceFieldName, emptyList());
            return this;
        }

        /**
         * Добавление элемента пути сразу за индексом списка.
         * <p>
         * Например, исходный путь: {@code banner.calloutIds[0]},
         * а в результате хочется получить: {@code banner.callouts[0].id}.
         * Для конвертации {@code calloutIds} в {@code callouts} используется обычный {@link #replace},
         * а для добавления {@code .id} используется данный метод:
         * <pre>
         *     .replace("calloutIds", "callouts")
         *     // в качестве имени списка передается имя до конвертации
         *     .appendListItems("calloutIds", "id") // в качестве имени списка передается имя до конвертации
         * </pre>
         *
         * @param listFieldName имя поля, в котором хранится список, до конвертации
         * @param appendingItem имя дополнительного элемента пути, который
         *                      будет подставляться сразу после индексов элементов списка
         */
        public MappingPathNodeConverter.Builder appendListItems(String listFieldName, String appendingItem) {
            checkArgument(!listItemDict.containsKey(listFieldName), "dictionary already contains %s", listFieldName);
            listItemDict.put(listFieldName, singletonList(appendingItem));
            return this;
        }

        /**
         * Добавление нескольких элементов пути сразу за индексом списка.
         * <p>
         * Например, исходный путь: {@code banner.calloutIds[0]},
         * а в результате хочется получить: {@code banner.callouts[0].data.id}.
         * Для конвертации {@code calloutIds} в {@code callouts} используется обычный {@link #replace},
         * а для добавления {@code .data.id} используется данный метод:
         * <pre>
         *     .replace("calloutIds", "callouts")
         *     // в качестве имени списка передается имя до конвертации
         *     .appendListItems("calloutIds", asList("data", "id"))
         * </pre>
         *
         * @param listFieldName  имя поля, в котором хранится список, до конвертации
         * @param appendingItems список имен дополнительных элементов пути, которые
         *                       будут подставляться сразу после индексов элементов списка
         */
        public MappingPathNodeConverter.Builder appendListItems(String listFieldName, List<String> appendingItems) {
            checkArgument(!listItemDict.containsKey(listFieldName), "dictionary already contains %s", listFieldName);
            listItemDict.put(listFieldName, appendingItems);
            return this;
        }

        public MappingPathNodeConverter build() {
            Map<PathNode.Field, Path> singleItemDictTmp = EntryStream.of(singleItemDict)
                    .mapKeys(PathHelper::field)
                    .mapValues(names -> mapList(names, PathHelper::field))
                    .mapValues(Path::new)
                    .toMap();
            Map<PathNode.Field, Path> listItemDictTmp = EntryStream.of(listItemDict)
                    .mapKeys(PathHelper::field)
                    .mapValues(names -> mapList(names, PathHelper::field))
                    .mapValues(Path::new)
                    .toMap();
            return new MappingPathNodeConverter(name, ImmutableMap.copyOf(singleItemDictTmp),
                    ImmutableMap.copyOf(listItemDictTmp));
        }
    }
}
