package ru.yandex.direct.validation.result;

import java.util.List;
import java.util.Objects;

import javax.annotation.Nullable;

import com.google.common.collect.ImmutableList;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;

/**
 * Путь к полю, для значения которого есть ошибка/предупреждение.
 * <p>
 * Состоит из списка элементов пути {@link PathNode}.
 * Для преобразования путей следует использовать {@link PathConverter}. Это бывает необходимо в случае, когда модель
 * данных в интерфейсном модуле отличается от модели в исполняющем коде.
 *
 * @see PathConverter
 * @see MappingPathConverter
 * @see PathNode
 */
public final class Path {
    private final List<PathNode> nodes;

    public Path(List<? extends PathNode> nodes) {
        if (nodes == null) {
            throw new IllegalArgumentException("nodes list is required");
        }
        this.nodes = ImmutableList.copyOf(nodes);
    }

    public List<PathNode> getNodes() {
        return nodes;
    }

    public boolean isEmpty() {
        return nodes.isEmpty();
    }

    /**
     * Проверить начинается ли путь c {@code subPath}
     */
    public boolean startsWith(Path subPath) {
        if (subPath.isEmpty()) {
            return true;
        }
        List<PathNode> patternNodes = subPath.getNodes();
        if (patternNodes.size() > this.getNodes().size()) {
            return false;
        }
        return EntryStream.zip(patternNodes, this.getNodes()
                .subList(0, patternNodes.size()))
                .allMatch(e -> e.getKey().equals(e.getValue()));
    }

    @Nullable
    public String getFieldName() {
        PathNode node = getLastFieldNode();
        return node == null ? null : node.toString();
    }

    private PathNode getLastFieldNode() {
        return StreamEx.of(nodes)
                .filter(node -> node instanceof PathNode.Field)
                .reduce((a, b) -> b)
                .orElse(null);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        Path path = (Path) o;
        return Objects.equals(nodes, path.nodes);
    }

    @Override
    public int hashCode() {
        return Objects.hash(nodes);
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        nodes.forEach(node -> node.appendTo(sb));
        return sb.toString();
    }
}
