package ru.yandex.direct.validation.result;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collector;

import kotlin.reflect.KProperty;

import ru.yandex.direct.model.ModelProperty;

import static com.google.common.base.Preconditions.checkArgument;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Вспомогательный класс для построения путей и элементов путей.
 * Пример использования: {@code Path path = path(field("banners"), index(3), field("id"))}
 */
public final class PathHelper {
    private PathHelper() {
    }

    public static Path path(PathNode... nodes) {
        return new Path(Arrays.asList(nodes));
    }

    public static Path pathFromStrings(String... names) {
        List<String> list = Arrays.asList(names);
        List<PathNode> fields = mapList(list, PathHelper::field);
        return path(fields);
    }

    public static Path emptyPath() {
        return new Path(Collections.emptyList());
    }

    public static Path path(List<PathNode> nodes) {
        return new Path(nodes);
    }

    public static Path concat(PathNode pathNode, Path path) {
        checkArgument(pathNode != null, "first pathNode is required");
        checkArgument(path != null, "path end is required");
        ArrayList<PathNode> res = new ArrayList<>();
        res.add(pathNode);
        res.addAll(path.getNodes());
        return new Path(res);
    }

    public static Path concat(Path startPath, Path endPath) {
        checkArgument(startPath != null, "start path is required");
        checkArgument(endPath != null, "end path is required");
        ArrayList<PathNode> res = new ArrayList<>();
        res.addAll(startPath.getNodes());
        res.addAll(endPath.getNodes());
        return new Path(res);
    }

    public static Path concat(Path path, PathNode pathNode) {
        checkArgument(path != null, "start path is required");
        checkArgument(pathNode != null, "last pathNode is required");
        List<PathNode> newPathNodes = new ArrayList<>(path.getNodes().size() + 1);
        newPathNodes.addAll(path.getNodes());
        newPathNodes.add(pathNode);
        return new Path(newPathNodes);
    }

    public static <T> PathNode.Field field(KProperty<T> property) {
        return field(property.getName());
    }

    public static PathNode.Field field(String name) {
        return new PathNode.Field(name);
    }

    public static PathNode.Field field(ModelProperty property) {
        return field(property.name());
    }

    public static PathNode.Index index(Integer index) {
        return new PathNode.Index(index);
    }

    public static <N extends PathNode> Collector<N, List<PathNode>, Path> toPath() {
        return Collector.of(
                ArrayList::new,
                List::add,
                (nodes1, nodes2) -> {
                    nodes1.addAll(nodes2);
                    return nodes1;
                },
                Path::new
        );
    }
}
