package ru.yandex.direct.validation.result;

/**
 * Элемент пути {@link Path}.
 * Может быть именем поля или индексом коллекции.
 * Умеет дописывать себя как часть пути в {@link StringBuilder}: метод {@link #appendTo(StringBuilder)}
 *
 * @see PathConverter
 * @see Path
 */
public interface PathNode {

    void appendTo(StringBuilder stringBuilder);

    /**
     * Элемент пути - индекс массива/списка
     */
    class Index implements PathNode, Comparable<Index> {
        private final int index;

        public Index(int index) {
            if (index < 0) {
                throw new IllegalArgumentException("index must be a positive number");
            }
            this.index = index;
        }

        public Integer getIndex() {
            return index;
        }

        @Override
        public void appendTo(StringBuilder builder) {
            builder.append("[").append(index).append("]");
        }

        @Override
        public int compareTo(Index other) {
            return Integer.compare(this.index, other.index);
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            Index index1 = (Index) o;
            return index == index1.index;
        }

        @Override
        public int hashCode() {
            return index;
        }

        @Override
        public String toString() {
            return String.valueOf(index);
        }
    }

    /**
     * Элемент пути - имя
     */
    class Field implements PathNode {

        private final String name;

        public Field(String name) {
            String trimmedName = name != null ? name.trim() : "";
            if (trimmedName.length() == 0) {
                throw new IllegalArgumentException("field name must not be null or empty");
            }
            this.name = trimmedName;
        }

        public String getName() {
            return name;
        }

        @Override
        public void appendTo(StringBuilder builder) {
            if (builder.length() != 0) {
                builder.append(".");
            }
            builder.append(name);
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }

            Field field = (Field) o;

            return name.equals(field.name);

        }

        @Override
        public int hashCode() {
            return name.hashCode();
        }

        @Override
        public String toString() {
            return name;
        }
    }
}
