package ru.yandex.direct.validation.result;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;

import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.model.ModelWithId;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.singletonList;
import static ru.yandex.direct.validation.result.PathHelper.concat;
import static ru.yandex.direct.validation.result.PathHelper.field;
import static ru.yandex.direct.validation.result.PathHelper.index;
import static ru.yandex.direct.validation.result.PathHelper.path;

/**
 * Иерархический результат валидации.
 * <p>
 * Содержит валидируемое значение, ошибки и предупреждения для него,
 * а так же ассоциативный массив дочерних узлов - результатов валидации и их имен.
 * Класс не должен использоваться напрямую клиентами библиотеки,
 * см. {@link ru.yandex.direct.validation.builder.ItemValidationBuilder},
 * {@link ru.yandex.direct.validation.builder.ListValidationBuilder}.
 * <p>
 * Как {@code ValidationResult} соотносится с валидируемыми объектами.
 * Например, есть класс {@code Address}:
 * <pre> {@code
 *
 * public class Address {
 *     private String city;
 *     private Integer postcode;
 * }
 * }</pre>
 * <p>
 * Над объектами этого класса необходимо проводить следующие проверки:
 * <ul>
 * <li>1. город не равен null (проверка поля city)</li>
 * <li>2. город существует (проверка поля city)</li>
 * <li>3. почтовый индекс не равен null (проверка поля postcode)</li>
 * <li>4. почтовый индекс соответствует городу (проверка консистентности всего объекта)</li>
 * </ul>
 * <p>
 * Тогда при валидации объекта этого класса будет создан экземпляр
 * {@code ValidationResult<Address, Defect>}, который будет содержать:
 * <ul>
 * <li>в поле value: проверяемый объект класса {@code Address}</li>
 * <li>в поле errors: возможно, ошибку несоответствия city и postcode</li>
 * <li>в поле subResults: для пути "city" - ValidationResult для поля {@code city}
 * и {@code ValidationResult} для поля {@code postcode}, которые могут содержать
 * ошибки соответствующих полей</li>
 * </ul>
 *
 * @param <T> тип валидируемого объекта
 * @param <D> тип дефекта (для ошибок и предупреждений)
 */
@ParametersAreNonnullByDefault
public class ValidationResult<T, D> {

    private final T value;
    private final List<D> errors;
    private final List<D> warnings;
    private final Map<PathNode, ValidationResult<?, D>> subResults;

    public ValidationResult(@Nullable T value) {
        this(value, null, null, null);
    }

    public ValidationResult(@Nullable T value, @SuppressWarnings("unused") Class<D> clazz) {
        this(value, null, null, null);
    }

    public ValidationResult(@Nullable T value, @Nullable Collection<D> errors, @Nullable Collection<D> warnings) {
        this(value, errors, warnings, null);
    }

    public ValidationResult(ValidationResult<T, D> validationResult) {
        this.value = validationResult.value;
        this.errors = new ArrayList<>(validationResult.errors);
        this.warnings = new ArrayList<>(validationResult.warnings);
        this.subResults = EntryStream.of(validationResult.subResults)
                .mapValues((Function<ValidationResult<?, D>, ValidationResult<?, D>>) ValidationResult::new)
                .toCustomMap(HashMap::new);
    }

    public ValidationResult(
            @Nullable T value,
            @Nullable Collection<D> errors,
            @Nullable Collection<D> warnings,
            @Nullable Map<PathNode, ValidationResult<?, D>> subResults) {
        this.value = value;
        this.errors = new ArrayList<>(errors != null ? errors : emptyList());
        this.warnings = new ArrayList<>(warnings != null ? warnings : emptyList());
        this.subResults = new HashMap<>(subResults != null ? subResults : emptyMap());
    }

    /**
     * Создать результат валидации без ошибок
     *
     * @param value Валидируемое значение
     * @param <T>   Тип валидируемого объекта
     * @param <D>   Тип дефекта (для ошибок и предупреждений)
     */
    public static <T, D> ValidationResult<T, D> success(@Nullable T value) {
        return new ValidationResult<>(value);
    }

    /**
     * Создать результат валидации с заданной ошибкой
     *
     * @param value Валидируемое значение
     * @param error Ошибка валидации
     * @param <T>   Тип валидируемого объекта
     * @param <D>   Тип дефекта (для ошибок и предупреждений)
     */
    public static <T, D> ValidationResult<T, D> failed(@Nullable T value, D error) {
        return new ValidationResult<>(value, singletonList(error), emptyList());
    }

    public static <V, D> List<V> getValidItems(ValidationResult<? extends List<? extends V>, D>  massValidation) {
        return new ArrayList<>(internalGetValidItemsWithIndex(massValidation, true).values());
    }

    public static <V, D> List<V> getValidItemsWithoutWarnings(ValidationResult<List<V>, D> massValidation) {
        return new ArrayList<>(internalGetValidItemsWithIndex(massValidation, false).values());
    }

    public static <V, D> Set<V> getInvalidItems(ValidationResult<? extends List<? extends V>, D> massValidation) {
        return getNotValidItemsWithDefects(massValidation, true).keySet();
    }

    public static <V, D> Set<V> getInvalidItemsWithoutWarnings(
            ValidationResult<? extends List<? extends V>, D> massValidation) {
        return getNotValidItemsWithDefects(massValidation, false).keySet();
    }

    /**
     * Заменить в результате валидации исходный список моделей и результаты валидации этих моделей
     * исходными ModelChanges-ами на базе которых эти модели были построены
     */
    public static <M extends ModelWithId, D> ValidationResult<List<ModelChanges<M>>, D> replaceModelListByModelChangesList(
            ValidationResult<List<M>, D> modelValidationResult,
            List<ModelChanges<M>> modelChangesList) {
        checkArgument(modelValidationResult.getValue().size() == modelChangesList.size());

        @SuppressWarnings("unchecked")
        Map<PathNode, ValidationResult<?, D>> newSubResults = (Map) EntryStream.of(
                modelValidationResult.getSubResults())
                .mapToValue((idx, subResult) ->
                        new ValidationResult<>(
                                modelChangesList.get(((PathNode.Index) idx).getIndex()),
                                subResult.getErrors(),
                                subResult.getWarnings(),
                                subResult.getSubResults()))
                .toMap();

        return new ValidationResult<>(
                modelChangesList,
                modelValidationResult.getErrors(),
                modelValidationResult.getWarnings(),
                newSubResults);
    }

    public static <V, D> Map<Integer, V> getValidItemsWithIndex(ValidationResult<? extends List<? extends V>, D> massValidation) {
        return internalGetValidItemsWithIndex(massValidation, true);
    }

    private static <V, D> LinkedHashMap<Integer, V> internalGetValidItemsWithIndex(
            ValidationResult<? extends List<? extends V>, D> massValidation, boolean includeItemsWithWarnings) {
        List<? extends V> allItems = massValidation.getValue();
        LinkedHashMap<Integer, V> results = new LinkedHashMap<>(allItems.size());
        for (int i = 0; i < allItems.size(); i++) {
            ValidationResult<?, D> validation = massValidation.subResults.get(index(i));

            if (validation == null ||
                    (!validation.hasAnyErrors() && (includeItemsWithWarnings || !validation.hasAnyWarnings()))) {
                results.put(i, allItems.get(i));
            }
        }
        return results;
    }

    public static <V, D> HashMap<V, List<DefectInfo<D>>> getNotValidItemsWithDefects(
            ValidationResult<? extends List<? extends V>, D> massValidation, boolean includeItemsWithWarnings) {
        List<? extends V> allItems = massValidation.getValue();
        HashMap<V, List<DefectInfo<D>>> results = new HashMap<>(allItems.size());
        for (int i = 0; i < allItems.size(); i++) {
            ValidationResult<?, D> validation = massValidation.subResults.get(index(i));

            if (validation != null &&
                    (validation.hasAnyErrors() || (includeItemsWithWarnings && validation.hasAnyWarnings()))) {
                ArrayList<DefectInfo<D>> defectList = new ArrayList<>();
                defectList.addAll(validation.flattenErrors());
                defectList.addAll(validation.flattenWarnings());
                results.put(allItems.get(i), defectList);
            }
        }
        return results;
    }

    /**
     * Объединить результаты валидации с успешно добавленными объектами
     *
     * @param validation      Объединенный результат валидации
     * @param successfulItems Идентификаторы успешно добавленных объектов
     * @param convert         Функция извлечения идентификатора из объекта
     * @param <TV>            Тип модели
     * @param <TR>            Тип идентификаторов результата
     * @param <D>             Тип дефекта
     * @return Объединенный список всех идентификаторов (валидных и невалиданных) в исходном порядке
     */
    public static <TV, TR, D> List<TR> mergeSuccessfulAndInvalidItems(ValidationResult<List<TV>, D> validation,
                                                                      List<TR> successfulItems,
                                                                      Function<TV, TR> convert) {
        if (getValidItems(validation).size() != successfulItems.size()) {
            throw new IllegalArgumentException(
                    "Successful items size should be equal to valid items size (from validation result)");
        }

        List<TR> results = new ArrayList<>(validation.getValue().size());
        Iterator<TR> iteratorSuccessful = successfulItems.iterator();
        for (int i = 0; i < validation.getValue().size(); i++) {
            ValidationResult<?, D> childValidation = validation.getSubResults().get(index(i));
            if (childValidation != null && childValidation.hasAnyErrors()) {
                results.add(convert.apply(validation.getValue().get(i)));
            } else {
                results.add(iteratorSuccessful.next());
            }
        }

        return results;
    }

    /**
     * Переносит с одного дерева ValidationResult на новое дерево ValidationResult только те ноды,
     * у которых есть ошибки или предупреждения.
     * <p>
     * Ошибки и предупреждения верхнего уровня копируются безусловно.
     *
     * @param validationFrom результат валидации, с которого копируются ноды с ошибками и предупреждениями
     * @param validationTo   результат валидации, на который копируются ноды с ошибками и предупреждениями
     */
    public static <D> void transferSubNodesWithIssues(
            ValidationResult<?, D> validationFrom,
            ValidationResult<?, D> validationTo) {
        validationTo.getErrors().addAll(validationFrom.getErrors());
        validationTo.getWarnings().addAll(validationFrom.getWarnings());
        validationFrom.getSubResults().forEach((pathNode, subResult) -> {
            if (subResult.hasAnyErrors() || subResult.hasAnyWarnings()) {
                ValidationResult<?, D> newSubResult =
                        validationTo.getOrCreateSubValidationResult(pathNode, subResult.getValue());
                transferSubNodesWithIssues(subResult, newSubResult);
            }
        });
    }

    public static <T extends List<?>, D> ValidationResult<?, D> getOrCreateSubValidationResultWithoutCast(
            ValidationResult<T, D> vr, int index) {
        return vr.getOrCreateSubValidationResultWithoutCast(new PathNode.Index(index), vr.getValue().get(index));
    }

    /**
     * @return валидируемое значение.
     */
    public T getValue() {
        return value;
    }

    /**
     * @param error ошибка для данного узла.
     */
    public ValidationResult<T, D> addError(D error) {
        errors.add(error);
        return this;
    }

    /**
     * @return ошибки текущего узла.
     */
    public List<D> getErrors() {
        return errors;
    }

    /**
     * @param warning предупреждение для текущего узла.
     */
    public void addWarning(D warning) {
        warnings.add(warning);
    }

    /**
     * @return предупреждения текущего узла.
     */
    public List<D> getWarnings() {
        return warnings;
    }

    /**
     * @return ассоциативный массив дочерних нод и их путей
     */
    public Map<PathNode, ValidationResult<?, D>> getSubResults() {
        return Collections.unmodifiableMap(subResults);
    }

    /**
     * Имеются ли ошибки на верхнем уровне этого результата валидации
     *
     * @return флаг наличия ошибок на верхнем уровне
     */
    public boolean hasErrors() {
        return !errors.isEmpty();
    }

    /**
     * Имеются ли предупреждения на верхнем уровне этого результата валидации
     *
     * @return флаг наличия предупреждений на верхнем уровне
     */
    public boolean hasWarnings() {
        return !warnings.isEmpty();
    }

    /**
     * @return true, если текущий результат валидации или любой дочерний узел
     * содержит хотя бы одну ошибку.
     */
    public boolean hasAnyErrors() {
        return hasErrors() || subResults.values().stream().anyMatch(ValidationResult::hasAnyErrors);
    }

    /**
     * @return true, если текущий результат валидации или любой дочерний узел
     * содержит хотя бы одно предупреждение.
     */
    public boolean hasAnyWarnings() {
        return hasWarnings() || subResults.values().stream().anyMatch(ValidationResult::hasAnyWarnings);
    }

    /**
     * Возвращает дочерний узел {@code ValidationResult} для указанного элемента пути (имени/индекса).
     *
     * @param pathNode имя/индекс, для которого необходимо получить дочерний узел {@code ValidationResult}.
     * @param value    значение, валидируемое в дочернем узле, используется либо для создания узла
     *                 при его отсутствии, либо для проверки соответствия значения в существующем узле
     *                 (переданная ссылка и ссылка на валидируемое значение в дочернем узле должны
     *                 ссылаться на один и тот же объект, в противном случае будет сгенерировано исключение!).
     * @param <V>      тип значения, валидируемого в дочернем узле.
     * @return дочерний узел (уже существующий или созданный на основе переданного валидируемого значения).
     */
    public <V> ValidationResult<?, D> getOrCreateSubValidationResultWithoutCast(PathNode pathNode, @Nullable V value) {
        ValidationResult<?, D> vr = subResults.computeIfAbsent(pathNode, pn -> new ValidationResult<>(value));
        if (vr.getValue() != value) {
            throw new IllegalArgumentException("attempt to call with other value than existing at passed path");
        }
        return vr;
    }

    @SuppressWarnings("unchecked")
    public <V> ValidationResult<V, D> getOrCreateSubValidationResult(PathNode pathNode, @Nullable V value) {
        return (ValidationResult<V, D>) getOrCreateSubValidationResultWithoutCast(pathNode, value);
    }

    public <M extends Model, V> ValidationResult<V, D> getOrCreateSubValidationResult(
            ModelProperty<M, V> property, @Nullable V value) {
        return getOrCreateSubValidationResult(field(property.name()), value);
    }

    @SuppressWarnings("unchecked")
    public ValidationResult<T, D> addSubResult(PathNode pathNode, ValidationResult vr) {
        if (subResults.containsKey(pathNode)) {
            throw new UnsupportedOperationException("Attempt to add already existing subresult to validation result");
        }

        subResults.put(pathNode, vr);
        return this;
    }

    /**
     * @return ошибки текущего узла и дочерних узлов в виде плоского списка {@link DefectInfo}
     */
    public List<DefectInfo<D>> flattenErrors() {
        return flattenErrors(null);
    }

    /**
     * @param converterProvider поставщик конвертеров пути
     */
    public List<DefectInfo<D>> flattenErrors(@Nullable PathNodeConverterProvider converterProvider) {
        return flattenDefects(ValidationResult::getErrors, null, null, path(), converterProvider);
    }

    /**
     * @return предупреждения текущего узла и дочерних узлов в виде плоского списка {@link DefectInfo}
     */
    public List<DefectInfo<D>> flattenWarnings() {
        return flattenWarnings(null);
    }

    /**
     * @param converterProvider поставщик конвертеров пути
     */
    public List<DefectInfo<D>> flattenWarnings(@Nullable PathNodeConverterProvider converterProvider) {
        return flattenDefects(ValidationResult::getWarnings, null, null, path(), converterProvider);
    }

    /**
     * Сливает текущий результат валидации с переданным в качестве параметра.
     *
     * @param vr дерево {@code ValidationResult}, ошибки/предупреждения которого
     *           нужно добавить в ошибки/предупреждения данного дерева.
     */
    public ValidationResult<T, D> merge(ValidationResult<?, D> vr) {
        if (vr.value != this.value) {
            throw new IllegalStateException("attempt to merge result with different value");
        }
        errors.addAll(vr.errors);
        warnings.addAll(vr.warnings);
        vr.subResults.forEach((pathNode, otherSubResult) -> {
            ValidationResult<?, D> subResult = subResults
                    .computeIfAbsent(pathNode, n -> new ValidationResult<>(otherSubResult.value));
            subResult.merge(otherSubResult);
        });

        return this;
    }

    @SuppressWarnings("unchecked")
    public <R> ValidationResult<R, D> transformUnchecked(ValidationResultTransformer<D> transformer) {
        return (ValidationResult<R, D>) transform(transformer);
    }

    public ValidationResult<?, D> transform(ValidationResultTransformer<D> transformer) {
        return transformer.transform(path(), this);
    }

    /**
     * Превращает дерево ошибок в плоский список
     *
     * @param defectProvider    ф-я получения дефектов с текущей ноды результата валидации (ошибки или ворнинги)
     * @param currentPath       текущий путь к ошибке
     * @param converterProvider поставщик конвертеров путей по классу родительского объекта
     * @return плоский список дефектов, обернутых в {@link DefectInfo}
     */
    private List<DefectInfo<D>> flattenDefects(
            Function<ValidationResult<?, D>, List<D>> defectProvider,
            @Nullable Class ownerClass, @Nullable PathNode.Field ownerPathNode,
            Path currentPath, @Nullable PathNodeConverterProvider converterProvider) {
        Class valueClass = getValueClass();

        List<DefectInfo<D>> defectInfoList = StreamEx.of(defectProvider.apply(this))
                .map(defect -> new DefectInfo<>(currentPath, value, defect))
                .toList();

        subResults.forEach((subPathNode, subResult) -> {

            checkState(subPathNode instanceof PathNode.Field || subPathNode instanceof PathNode.Index,
                    "path nodes must be either PathNode.Field or PathNode.Index");

            // для единичных объектов их родительским классом будет текущий класс
            // для элементов списка "родительским" классом будет класс, содержащий список
            Class ownerClassForSubNode = subPathNode instanceof PathNode.Field ? valueClass : ownerClass;

            PathNodeConverter converter = converterProvider != null && ownerClassForSubNode != null ?
                    converterProvider.getConverter(ownerClassForSubNode) : null;

            // для единичных объектов путь к ним будет текущим обрабатываемым полем
            // для элементов списка путь к ним будет не индексом, а полем класса, в котором лежит список
            PathNode.Field ownerPathNodeForSubNode = subPathNode instanceof PathNode.Field ?
                    (PathNode.Field) subPathNode : ownerPathNode;

            Path convertedPath;
            if (converter != null) {

                if (subPathNode instanceof PathNode.Field) {
                    convertedPath = converter.convert((PathNode.Field) subPathNode);
                    checkState(convertedPath != null, "converter returned null for field %s in class %s",
                            subPathNode, valueClass);
                } else {
                    convertedPath = converter.convert(ownerPathNodeForSubNode, (PathNode.Index) subPathNode);
                    checkState(convertedPath != null, "converter returned null for field %s[%s] in class %s",
                            ownerPathNode, subPathNode, ownerClass);
                }
            } else {
                convertedPath = path(subPathNode);
            }

            Path subPath = concat(currentPath, convertedPath);
            List<DefectInfo<D>> subResultDefectInfos = subResult.flattenDefects(defectProvider,
                    ownerClassForSubNode, ownerPathNodeForSubNode, subPath, converterProvider);
            defectInfoList.addAll(subResultDefectInfos);
        });
        return defectInfoList;
    }

    private Class getValueClass() {
        return this.value == null ? null : this.value.getClass();
    }

    @Override
    public String toString() {
        return "ValidationResult{" +
                "anyErrors=" + hasAnyErrors() +
                ", anyWarnings=" + hasAnyWarnings() +
                '}';
    }

    public interface ValidationResultTransformer<D> {

        default <OV> Object transformValue(Path path, @Nullable OV oldValue) {
            return oldValue;
        }

        default List<D> transformErrors(@SuppressWarnings("unused") Path path, List<D> errors) {
            return errors;
        }

        default List<D> transformWarnings(@SuppressWarnings("unused") Path path, List<D> warnings) {
            return warnings;
        }

        default ValidationResult<?, D> transform(Path path, ValidationResult<?, D> validationResult) {
            ValidationResult<?, D> transformedVr = new ValidationResult<>(transformValue(path, validationResult.value));

            List<D> transformedErrors = transformErrors(path, validationResult.getErrors());
            transformedVr.errors.addAll(transformedErrors);

            List<D> transformedWarnings = transformWarnings(path, validationResult.getWarnings());
            transformedVr.warnings.addAll(transformedWarnings);

            Map<PathNode, ValidationResult<?, D>> transformedSubResults =
                    transformSubResults(path, validationResult.subResults);
            transformedVr.subResults.putAll(transformedSubResults);

            return transformedVr;
        }

        default Map<PathNode, ValidationResult<?, D>> transformSubResults(Path path,
                                                                          Map<PathNode, ValidationResult<?, D>> subResults) {
            Map<PathNode, ValidationResult<?, D>> transformedSubResults = new HashMap<>();
            subResults.forEach(
                    (pathNode, currentSubVr) -> {
                        Path subPath = concat(path, pathNode);
                        transformedSubResults.put(pathNode, transform(subPath, currentSubVr));
                    });
            return transformedSubResults;
        }
    }
}
