package ru.yandex.direct.validation.result;

import java.util.Collection;
import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;

import static com.google.common.base.Preconditions.checkArgument;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.validation.result.PathHelper.index;

@ParametersAreNonnullByDefault
public class ValidationResultUtils {
    private ValidationResultUtils() {
    }

    public static ValidationResult mergeSimilarListValidationResults(Collection<ValidationResult> validationResults) {
        List<Map.Entry<Object, ValidationResult>> mergedList = StreamEx.of(validationResults)
                .flatMap(ValidationResultUtils::indexSubresultsByValues)
                .toList();

        var topLevelErrors = StreamEx.of(validationResults)
                .flatMap(vr -> nvl(vr.getErrors(), List.of()).stream())
                .toList();
        var topLevelWarnings = StreamEx.of(validationResults)
                .flatMap(vr -> nvl(vr.getWarnings(), List.of()).stream())
                .toList();

        var listValue = StreamEx.of(mergedList).map(Map.Entry::getKey).toList();
        @SuppressWarnings("unchecked")
        var singleValidationResult = new ValidationResult(listValue, topLevelErrors, topLevelWarnings);

        EntryStream.of(mergedList)
                .mapValues(Map.Entry::getValue)
                .nonNullValues()
                .forEach(e -> singleValidationResult.addSubResult(index(e.getKey()), e.getValue()));

        return singleValidationResult;
    }

    private static EntryStream<Object, ValidationResult> indexSubresultsByValues(ValidationResult vr) {
        @SuppressWarnings("unchecked")
        List<Object> values = (List<Object>) vr.getValue();
        return EntryStream.of(values)
                .invert()
                .mapValues(i -> (ValidationResult) vr.getSubResults().get(index(i)));
    }

    public static <T, D> List<ValidationResult<T, D>> splitValidationResult(
            List<T> items, ValidationResult<List<T>, D> validationResult) {
        return EntryStream.of(items)
                .mapKeyValue((idx, item) -> validationResult.getOrCreateSubValidationResult(index(idx), item))
                .toList();
    }

    public static <T, D> ValidationResult<List<T>, D> mergeValidationResults(
            List<T> items, List<? extends ValidationResult<? extends T, D>> validationResults) {
        checkArgument(items.size() == validationResults.size());
        ValidationResult<List<T>, D> vr = new ValidationResult<>(items);
        EntryStream.of(items)
                .forKeyValue((idx, item) -> vr.addSubResult(index(idx), validationResults.get(idx)));
        return vr;
    }

    public static <T, D> boolean hasAnyErrors(List<? extends ValidationResult<? extends T, D>> validationResultList) {
        return StreamEx.of(validationResultList).anyMatch(ValidationResult::hasAnyErrors);
    }
}
