package ru.yandex.direct.validation.util;

import java.util.List;
import java.util.Set;
import java.util.function.Supplier;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.model.ModelWithId;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.maxListSize;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.minListSize;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.unique;
import static ru.yandex.direct.validation.constraint.CommonConstraints.inSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;
import static ru.yandex.direct.validation.defect.CollectionDefects.duplicatedElement;
import static ru.yandex.direct.validation.defect.CommonDefects.objectNotFound;

/**
 * Инструмент, упрощающий предварительную валидацию списка ModelChanges для операции обновления.
 * Проводит базовые для всех операций обновления проверки: количество изменений (min/max),
 * отсутствие элементов с id == null, отсутствие дубликатов по id, существование обновляемых элементов.
 * <p>
 * Если проверка кол-ва не нужна, её можно пропустить, передав вметсо значаний minSize/maxSize null
 * <p>
 * После проведения базовой валидация с помощью этого инструмента,
 * можно легко провести дополнительные проверки на полученном объекте
 * {@code ValidationResult<List<ModelChanges<T>>, Defect>.
 *
 * @param <T> тип модели.
 */
@ParametersAreNonnullByDefault
public class ModelChangesValidationTool {

    @Nullable
    private final Integer minSize;
    @Nullable
    private final Integer maxSize;

    private final Defect duplicatedItemDefect;

    private final Defect objectNotFoundDefect;

    public ModelChangesValidationTool() {
        this(builder());
    }

    private ModelChangesValidationTool(Builder b) {
        this.minSize = b.minSize;
        this.maxSize = b.maxSize;
        this.duplicatedItemDefect = nvl(b.duplicatedItemDefect, duplicatedElement());
        this.objectNotFoundDefect = nvl(b.objectNotFoundDefect, objectNotFound());
    }

    /**
     * @return Билдер без обязательных полей.
     */
    public static Builder builder() {
        return new Builder();
    }

    /**
     * Проводит базовые проверки списка ModelChanges для операций обновления.
     *
     * @param modelChangesList список ModelChanges
     * @param existingIds      существующие id объектов для проверки существования указанных объектов.
     * @return результат валидации списка ModelChanges.
     */
    public <T extends ModelWithId> ValidationResult<List<ModelChanges<T>>, Defect> validateModelChangesList(
            List<ModelChanges<T>> modelChangesList, Set<Long> existingIds) {
        ValidationResult<List<ModelChanges<T>>, Defect> vr = new ValidationResult<>(modelChangesList, Defect.class);
        validateModelChangesList(vr, () -> existingIds);
        return vr;
    }

    /**
     * Проводит базовые проверки списка ModelChanges для операций обновления.
     *
     * @param preValidationResult результаты предварительной валидации
     * @param existingIds         существующие id объектов для проверки существования указанных объектов.
     * @return результат валидации списка ModelChanges.
     */
    public <T extends ModelWithId> void validateModelChangesList(
            ValidationResult<List<ModelChanges<T>>, Defect> preValidationResult,
            Supplier<Set<Long>> existingIds) {
        new ListValidationBuilder<>(preValidationResult)
                .checkBy(this::validateOperation)
                .checkBy(elements -> validateElements(elements, existingIds), When.isValid());
    }

    private <T extends ModelWithId> ValidationResult<List<ModelChanges<T>>, Defect> validateOperation(
            List<ModelChanges<T>> modelChangesList) {
        ListValidationBuilder<ModelChanges<T>, Defect> vb = ListValidationBuilder.of(
                modelChangesList, Defect.class);
        if (minSize != null) {
            vb.check(minListSize(minSize));
        }
        if (maxSize != null) {
            vb.check(maxListSize(maxSize));
        }
        return vb.getResult();
    }

    private <T extends ModelWithId> ValidationResult<List<ModelChanges<T>>, Defect> validateElements(
            List<ModelChanges<T>> modelChangesList, Supplier<Set<Long>> existingIdsSupplier) {
        return ListValidationBuilder.of(modelChangesList, Defect.class)
                .checkEach(notNull())
                .checkEach(unique(ModelChanges::getId), duplicatedItemDefect)
                .checkEachBy(element -> validateModelChanges(element, existingIdsSupplier), When.isValid())
                .getResult();
    }

    private <T extends ModelWithId> ValidationResult<ModelChanges<T>, Defect> validateModelChanges(
            ModelChanges<T> modelChanges, Supplier<Set<Long>> existingIdsSupplier) {
        ItemValidationBuilder<ModelChanges<T>, Defect> vb =
                ItemValidationBuilder.of(modelChanges);

        vb.item(modelChanges.getId(), "id")
                .check(notNull())
                .check(validId())
                .check(inSet(existingIdsSupplier), objectNotFoundDefect, When.isValid());

        return vb.getResult();
    }

    public static final class Builder {
        private Integer minSize;
        private Integer maxSize;
        private Defect duplicatedItemDefect;
        private Defect objectNotFoundDefect;

        private Builder() {
        }

        /**
         * @param minSize минимально допустимый размер списка ModelChanges
         */
        public Builder minSize(Integer minSize) {
            this.minSize = minSize;
            return this;
        }

        /**
         * @param maxSize максимально допустимый размер списка ModelChanges
         */
        public Builder maxSize(Integer maxSize) {
            this.maxSize = maxSize;
            return this;
        }

        /**
         * @param duplicatedItemDefect тип дефекта для дублирующихся id
         */
        public Builder duplicatedItemDefect(Defect duplicatedItemDefect) {
            this.duplicatedItemDefect = duplicatedItemDefect;
            return this;
        }

        /**
         * @param objectNotFoundDefect тип дефекта для ненайденного объекта
         */
        public Builder objectNotFoundDefect(Defect objectNotFoundDefect) {
            this.objectNotFoundDefect = objectNotFoundDefect;
            return this;
        }

        public ModelChangesValidationTool build() {
            return new ModelChangesValidationTool(this);
        }
    }
}
