package ru.yandex.direct.validation.util

import ru.yandex.direct.model.Model
import ru.yandex.direct.model.ModelChanges
import ru.yandex.direct.model.ModelProperty
import ru.yandex.direct.model.ModelWithId
import ru.yandex.direct.validation.builder.Constraint
import ru.yandex.direct.validation.builder.ItemValidationBuilder
import ru.yandex.direct.validation.builder.ListValidationBuilder
import ru.yandex.direct.validation.builder.When
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult
import ru.yandex.direct.validation.wrapper.ModelChangesValidationBuilder
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder
import java.util.function.Predicate
import kotlin.reflect.KProperty0
import kotlin.reflect.KProperty1

/*
 * Простые хелперы для удобного написания валидации на Kotlin (без ModelPropery)
 *
 * Пример использования:
 *
 * val result = validateObject(obj) {
 *     property(Obj::id) {
 *         check(CommonConstraints.notNull())
 *         check(NumberConstraints.lessThan(100))
 *         check(NumberConstraints.greaterThan(10))
 *     }
 *
 *     property(obj::name)
 *         .check(CommonConstraints.notNull())
 * }
 */

typealias D = Defect<*>

fun <T> validateObject(item: T, block: ItemValidationBuilder<T, D>.() -> Unit): ValidationResult<T, D> =
    ItemValidationBuilder.of<T, D>(item).apply(block).result

fun <T> validateObject(
    vr: ValidationResult<T, D>,
    block: ItemValidationBuilder<T, D>.() -> Unit
): ValidationResult<T, D> = ItemValidationBuilder(vr).apply(block).result

fun <T> validateList(list: List<T>?, block: ListValidationBuilder<T, D>.() -> Unit): ValidationResult<List<T>, D> =
    ListValidationBuilder.of<T, D>(list).apply(block).result

fun <T> validateList(
    vr: ValidationResult<List<T>, D>,
    block: ListValidationBuilder<T, D>.() -> Unit
): ValidationResult<List<T>, D> = ListValidationBuilder(vr).apply(block).result

fun <M : Model> validateModel(item: M, block: ModelItemValidationBuilder<M>.() -> Unit): ValidationResult<M, D> =
    ModelItemValidationBuilder.of(item).apply(block).result

fun <M : ModelWithId> validateModelChanges(
    mc: ModelChanges<M>,
    block: ModelChangesValidationBuilder<M>.() -> Unit
): ValidationResult<ModelChanges<M>, D> = ModelChangesValidationBuilder.of(mc).apply(block).result

// object attributes
fun <T, D, V> ItemValidationBuilder<T, D>.property(prop: KProperty1<T, V>): ItemValidationBuilder<V, D> =
    this.item(prop.get(this.result.value), prop.name)

fun <T, D, V> ItemValidationBuilder<T, D>.property(prop: KProperty0<V>): ItemValidationBuilder<V, D> =
    this.item(prop.get(), prop.name)

fun <T, D, V> ItemValidationBuilder<T, D>.property(
    prop: KProperty1<in T, V>,
    init: ItemValidationBuilder<V, D>.() -> Unit
) = this.item(prop.get(this.result.value), prop.name).init()

fun <T, D, V> ItemValidationBuilder<T, D>.property(
    prop: KProperty0<V>,
    init: ItemValidationBuilder<V, D>.() -> Unit
) = this.item(prop.get(), prop.name).init()

// model validation
fun <M : Model, V> ModelItemValidationBuilder<M>.item(
    prop: ModelProperty<in M, V>,
    block: ItemValidationBuilder<V, D>.() -> Unit,
) = this.item(prop).block()

fun <M : Model, V : Model> ModelItemValidationBuilder<M>.modelItem(
    prop: ModelProperty<in M, V>,
    block: ModelItemValidationBuilder<V>.() -> Unit,
) = this.modelItem(prop).block()

fun <M : Model, V> ModelItemValidationBuilder<M>.list(
    prop: ModelProperty<in M, List<V>?>,
    block: ListValidationBuilder<V, D>.() -> Unit,
) = this.list(prop).block()

fun <M : ModelWithId, V> ModelChangesValidationBuilder<M>.item(
    prop: ModelProperty<in M, V>,
    block: ModelChangesValidationBuilder<M>.ModelChangesValidationBuilderInner<V>.() -> Unit,
) = this.item(prop).block()

// for list attributes
fun <T, D, V> ItemValidationBuilder<T, D>.listProperty(prop: KProperty1<T, List<V>?>): ListValidationBuilder<V, D> =
    this.list(prop.get(this.result.value), prop.name)

fun <T, D, V> ItemValidationBuilder<T, D>.listProperty(prop: KProperty0<List<V>?>): ListValidationBuilder<V, D> =
    this.list(prop.get(), prop.name)

fun <T, D, V> ItemValidationBuilder<T, D>.listProperty(
    prop: KProperty1<T, List<V>?>,
    init: ListValidationBuilder<V, D>.() -> Unit
) = this.list(prop.get(this.result.value), prop.name).init()

fun <T, D, V> ItemValidationBuilder<T, D>.listProperty(
    prop: KProperty0<List<V>?>,
    init: ListValidationBuilder<V, D>.() -> Unit
) = this.list(prop.get(), prop.name).init()

// easier checks
fun <T, D> ItemValidationBuilder<T, D>.check(defect: D, predicate: Predicate<T>): ItemValidationBuilder<T, D> =
    this.check(Constraint.fromPredicate(predicate, defect))

fun <T, D> ListValidationBuilder<T, D>.check(defect: D, predicate: Predicate<List<T>>): ListValidationBuilder<T, D> =
    this.check(Constraint.fromPredicate(predicate, defect))

fun <T, D> ItemValidationBuilder<T, D>.check(defect: D, `when`: When<T, D>, predicate: Predicate<T>): ItemValidationBuilder<T, D> =
    this.check(Constraint.fromPredicate(predicate, defect), `when`)

fun <T, D> ListValidationBuilder<T, D>.check(defect: D, `when`: When<List<T>, D>, predicate: Predicate<List<T>>): ListValidationBuilder<T, D> =
    this.check(Constraint.fromPredicate(predicate, defect), `when`)

fun <T, D> ListValidationBuilder<T, D>.checkEach(defect: D, predicate: Predicate<T>): ListValidationBuilder<T, D> =
    this.checkEach(Constraint.fromPredicate(predicate, defect))
