package ru.yandex.direct.validation.util;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;

import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.model.ModelWithId;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.PathNode;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.result.ValidationResult.getValidItems;
import static ru.yandex.direct.validation.result.ValidationResult.transferSubNodesWithIssues;

public final class ValidationUtils {

    private ValidationUtils() {
    }

    /**
     * Преобразует результат валидации списка ModelChanges ({@code ValidationResult<List<ModelChanges<M>, D>})
     * в результат валидации списка моделей ({@code ValidationResult<List<M>, D>}). Все ошибки
     * каждого ModelChanges переносятся в результат валидации соответствующих моделей (по id).
     * <p>
     * Список моделей формируется следующим образом: для каждого ModelChanges из входного списка ищется
     * модель в передаваемой коллекции моделей по id, а если не находится, то вместо нее создается заглушка.
     * Таким образом, размер выходного списка соответствует размеру входного, и каждая модель выходного
     * списка соответствует ModelChanges из входного по id.
     * <p>
     * Стандартный кейс использования метода: передать модели только для валидных ModelChanges,
     * тогда для ошибочных ModelChanges будут созданы заглушки, которые не подлежат дальнейшей валидации.
     * На тех узлах ValidationResult, на которых будут заглушки, так же будут и ошибки. Избежать валидации
     * заглушек с ошибками можно с помощью {@link When#isValid()}.
     *
     * @param modelChangesValidationResult результат валидации списка ModelChanges.
     * @param models                       список моделей
     * @param modelStubCreator             создатель заглушек моделей
     * @param <T>                          тип моделей
     * @return результат валидации списка моделей
     */
    public static <T extends ModelWithId, D> ValidationResult<List<T>, D> modelChangesValidationToModelValidation(
            ValidationResult<List<ModelChanges<T>>, D> modelChangesValidationResult,
            Collection<T> models, Function<Long, T> modelStubCreator) {
        Map<Long, T> idToModel = listToMap(models, ModelWithId::getId);
        // список моделей для нового результата валидации, в котором элементы -
        // это либо заполненная модель, либо заглушка с выставленным id
        List<T> computedModels = mapList(modelChangesValidationResult.getValue(),
                modelChanges -> idToModel.computeIfAbsent(modelChanges.getId(), modelStubCreator));

        return convertToValidationResult(modelChangesValidationResult, computedModels);
    }

    /**
     * Дублируется поведение метода modelChangesValidationToModelValidation с wildcard в типах аргументов
     * todo ssdmitriev: DIRECT-100166: избавиться от дубликатов
     */
    public static <T extends ModelWithId, D> ValidationResult<? extends List<? extends T>, D> modelChangesValidationToModelValidationForSubtypes(
            ValidationResult<? extends List<? extends ModelChanges<? extends T>>, D> modelChangesValidationResult,
            Collection<? extends T> models, Function<Long, ? extends T> modelStubCreator) {
        Map<Long, T> idToModel = listToMap(models, ModelWithId::getId);
        // список моделей для нового результата валидации, в котором элементы -
        // это либо заполненная модель, либо заглушка с выставленным id
        List<T> computedModels = mapList(modelChangesValidationResult.getValue(),
                modelChanges -> idToModel.computeIfAbsent(modelChanges.getId(), modelStubCreator));
        return convertToValidationResultForSubtypes(modelChangesValidationResult, computedModels);
    }

    /**
     * Преобразование результата валидации одного типа к другому, для списка моделей.
     *
     * @param fromValidationResult исходный результат валидации
     * @param models               объекты, для которых создаётся новая валидация
     * @param <F>                  тип изначальных объектов валидации
     * @param <T>                  тип объектов валидации после преобразования
     * @return преобразованный результат валидации
     */
    public static <F, T, D> ValidationResult<List<T>, D> convertToValidationResult(
            ValidationResult<List<F>, D> fromValidationResult, List<T> models) {
        ValidationResult<List<T>, D> validationResult = new ValidationResult<>(models);
        validationResult.getErrors().addAll(fromValidationResult.getErrors());
        validationResult.getWarnings().addAll(fromValidationResult.getWarnings());

        fromValidationResult.getSubResults().forEach((pathNode, subResultFrom) -> {
            int idx = ((PathNode.Index) pathNode).getIndex();
            T validatedModel = models.get(idx);
            ValidationResult<?, D> subResultTo =
                    validationResult.getOrCreateSubValidationResult(pathNode, validatedModel);
            transferIssuesFromValidationToValidation(subResultFrom, subResultTo);
        });

        return validationResult;
    }

    /**
     * Преобразование результата валидации одного типа к другому, для одной модели.
     *
     * @param fromValidationResult исходный результат валидации
     * @param model                объект, для которого создаётся новая валидация
     * @param <F>                  тип изначальных объектов валидации
     * @param <T>                  тип объектов валидации после преобразования
     * @return преобразованный результат валидации
     */
    public static <F, T, D> ValidationResult<T, D> convertToValidationResult(
            ValidationResult<F, D> fromValidationResult, T model) {
        ValidationResult<T, D> validationResult = new ValidationResult<>(model);
        validationResult.getErrors().addAll(fromValidationResult.getErrors());
        validationResult.getWarnings().addAll(fromValidationResult.getWarnings());

        fromValidationResult.getSubResults().forEach((pathNode, subResultFrom) -> {
            ValidationResult<?, D> subResultTo =
                    validationResult.getOrCreateSubValidationResult(pathNode, model);
            transferIssuesFromValidationToValidation(subResultFrom, subResultTo);
        });

        return validationResult;
    }

    public static <F, T, D> ValidationResult<? extends List<? extends T>, D> convertToValidationResultForSubtypes(
            ValidationResult<? extends List<? extends F>, D> fromValidationResult, List<? extends T> models) {

        ValidationResult<List<? extends T>, D> validationResult = new ValidationResult<>(models);
        validationResult.getErrors().addAll(fromValidationResult.getErrors());
        validationResult.getWarnings().addAll(fromValidationResult.getWarnings());
        fromValidationResult.getSubResults().forEach((pathNode, subResultFrom) -> {
            int idx = ((PathNode.Index) pathNode).getIndex();
            T validatedModel = models.get(idx);
            ValidationResult<?, D> subResultTo =
                    validationResult.getOrCreateSubValidationResult(pathNode, validatedModel);
            transferIssuesFromValidationToValidation(subResultFrom, subResultTo);
        });
        return validationResult;
    }

    /**
     * Преобразует результат валидации списка одного типа <F> в результат валидации списка с другим типом <T>.
     * Все ошибки переносятся в результат валидации на верхний уровень полученного объекта
     *
     * @param validationResultFrom результат валидации, который нужно преобразовать
     * @param converter            преобразование модели типа <F> в <T>
     * @param <F>                  тип модели валидации из которого конвертировать
     * @param <T>                  тип модели валидации в который конвертировать
     * @return результат валидации списка с новым типом
     */
    public static <F, T, D> ValidationResult<List<T>, D> flattenValidationResult(
            ValidationResult<List<F>, D> validationResultFrom, Function<F, T> converter) {
        List<T> objects = mapList(validationResultFrom.getValue(), converter);
        ValidationResult<List<T>, D> newValidationResult = new ValidationResult<>(objects);
        newValidationResult.getErrors().addAll(validationResultFrom.getErrors());
        newValidationResult.getWarnings().addAll(validationResultFrom.getWarnings());

        validationResultFrom.getSubResults().forEach((pathNode, subResultFrom) -> {
            int idx = ((PathNode.Index) pathNode).getIndex();
            T validatedObject = objects.get(idx);
            ValidationResult<?, D> subResultTo =
                    newValidationResult.getOrCreateSubValidationResult(pathNode, validatedObject);
            transferIssuesFromValidationToTopLevel(subResultFrom, subResultTo);
        });

        return newValidationResult;
    }

    /**
     * Извлекает из переданной коллекции AppliedChanges только те объекты, которые соответствуют
     * успешно провалидированным моделям, находящимся в результате валидации (по их id).
     * <p>
     * Стандартный кейс использования:
     * 1. из базы достаются модели (получается коллекция моделей);
     * 2. к ним применяются изменения (получается коллекция {@code AppliedChanges});
     * 3. из коллекции {@code AppliedChanges} извлекаются и валидируются модели
     * (получается {@code ValidationResult<List<M>, D>});
     * 4. из списка {@code AppliedChanges} с помощью данного метода извлекаются только валидные;
     * 5. вызывается метод репозитория с валидными AppliedChanges.
     *
     * @param appliedChangesList список объектов AppliedChanges
     * @param validationResult   результат валидации списка моделей
     * @param <M>                тип модели
     * @return отображение индекс элемента -> AppliedChanges, соответствующее успешно провалидированным моделям.
     */
    public static <M extends ModelWithId, D> Map<Integer, AppliedChanges<M>> extractOnlyValidAppliedChangesWithIndex(
            Map<Integer, AppliedChanges<M>> appliedChangesList,
            ValidationResult<List<M>, D> validationResult) {
        Set<Long> idsOfValidModels = StreamEx.of(getValidItems(validationResult))
                .map(ModelWithId::getId)
                .toSet();
        return EntryStream.of(appliedChangesList)
                .filterValues(v -> idsOfValidModels.contains(v.getModel().getId()))
                .toMap();
    }


    /**
     * Извлекает из переданной коллекции AppliedChanges только те объекты, которые соответствуют
     * успешно провалидированным моделям, находящимся в результате валидации (по их id).
     */
    public static <M extends ModelWithId, D> Collection<AppliedChanges<M>> extractOnlyValidAppliedChanges(
            Collection<AppliedChanges<M>> appliedChangesList,
            ValidationResult<List<M>, D> validationResult) {
        Set<Long> idsOfValidModels = StreamEx.of(getValidItems(validationResult))
                .map(ModelWithId::getId)
                .toSet();
        return filterList(appliedChangesList,
                appliedChanges -> idsOfValidModels.contains(appliedChanges.getModel().getId()));
    }

    /**
     * Переносит ошибки и предупреждения с одного результата валидации на другой, включая дочерние
     *
     * @param validationFrom результат, с которого берутся ошибки и предупреждения
     * @param validationTo   результат, на который переносятся ошибки и предупреждения
     */
    public static <D> void transferIssuesFromValidationToValidation(ValidationResult<?, D> validationFrom,
                                                                    ValidationResult<?, D> validationTo) {
        validationTo.getErrors().addAll(validationFrom.getErrors());
        validationTo.getWarnings().addAll(validationFrom.getWarnings());
        validationFrom.getSubResults().forEach((pathNode, subResultFrom) -> {
            ValidationResult<?, D> subResultTo =
                    validationTo.getOrCreateSubValidationResult(pathNode, subResultFrom.getValue());
            transferIssuesFromValidationToValidation(subResultFrom, subResultTo);
        });
    }


    public static <D> void transferIssuesFromValidationToValidationWithNewValue(
            ValidationResult<?, D> validationFrom,
            ValidationResult<?, D> validationTo) {
        validationTo.getErrors().addAll(validationFrom.getErrors());
        validationTo.getWarnings().addAll(validationFrom.getWarnings());
        validationFrom.getSubResults().forEach((pathNode, subResultFrom) -> {
            ValidationResult<?, D> subResultTo = validationTo.getSubResults().get(pathNode);

            Object value = subResultTo != null ? subResultTo.getValue() : subResultFrom.getValue();
            subResultTo = validationTo.getOrCreateSubValidationResult(pathNode, value);

            transferIssuesFromValidationToValidationWithNewValue(subResultFrom, subResultTo);
        });
    }


    /**
     * Переносит ошибки и предупреждения с одного результата валидации на другой, включая дочерние, на верхний уровень
     *
     * @param validationFrom результат, с которого берутся ошибки и предупреждения
     * @param validationTo   результат, на который переносятся ошибки и предупреждения (на верхний уровень)
     */
    public static <D> void transferIssuesFromValidationToTopLevel(ValidationResult<?, D> validationFrom,
                                                                  ValidationResult<?, D> validationTo) {
        validationTo.getErrors().addAll(validationFrom.getErrors());
        validationTo.getWarnings().addAll(validationFrom.getWarnings());
        validationFrom.getSubResults().forEach((pathNode, subResultFrom)
                -> transferIssuesFromValidationToTopLevel(subResultFrom, validationTo));
    }

    /**
     * Копирует результат валидации объекта, перенося на него только те саб-ноды,
     * в которых содержатся ошибки или предупреждения.
     * <p>
     * Нода верхнего уровня создается безусловно, т.е. метод никогда не возвращает null.
     * <p>
     * Очищая результат валидации от саб-нод без ошибок позволяет в дальнейшем модифицировать
     * успешно провалидированные поля/объекты, и провалидировать их снова, не нарушая структуру
     * результата валидации. При этом изменить и снова провалидировать те объекты, на которых есть ошибки,
     * не получится, так как в дереве остались их прежние значения, и
     * {@link ValidationResult#getOrCreateSubValidationResult(PathNode, Object)} сгенерирует исключение.
     *
     * @param validationResult исходный результат валидации
     * @param <T>              тип провалидироанного объекта
     * @param <D>              тип дефекта
     * @return копия результата валидации, содержащая только те саб-ноды, в которых есть ошибки и предупреждения
     */
    public static <T, D> ValidationResult<T, D> cloneValidationResultSubNodesWithIssues(
            ValidationResult<T, D> validationResult) {
        ValidationResult<T, D> newValidationResult = new ValidationResult<>(validationResult.getValue());
        transferSubNodesWithIssues(validationResult, newValidationResult);
        return newValidationResult;
    }
}
