package ru.yandex.direct.validation.wrapper;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.model.ModelWithId;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

/**
 * Данный ValidationBuilder перед запуском каждой проверки, проверяет возвращает ли isPropChanged() true и если нет,
 * то не выполняет проверку.
 *
 * <p>Property в ModelChanges нужно валидировать, только если есть что валидировать, т.е. было передано значение для
 * property, т.е. isPropChanged = true (при этом новое значение может совпадать с текущим или быть null). Таким образом
 * вызов isPropChanged обязателен и позволяет в том числе отличить ситуацию "передан null" от ситуации
 * "не передано ничего".
 */
@ParametersAreNonnullByDefault
public class ModelChangesValidationBuilder<M extends ModelWithId> extends
        ItemValidationBuilder<ModelChanges<M>, Defect> {

    private ModelChangesValidationBuilder(ValidationResult<ModelChanges<M>, Defect> validationResult) {
        super(validationResult);
    }

    public static <M extends ModelWithId> ModelChangesValidationBuilder<M> of(ModelChanges<M> changes) {
        return new ModelChangesValidationBuilder<>(new ValidationResult<>(changes));
    }

    public <V> ModelChangesValidationBuilderInner<V> item(ModelProperty<? super M, V> prop) {
        return new ModelChangesValidationBuilderInner<>(prop);
    }

    private <V> ItemValidationBuilder<V, Defect> itemInternal(ModelProperty<? super M, V> prop) {
        return item(getResult().getValue().getPropIfChanged(prop), prop.name());
    }

    public class ModelChangesValidationBuilderInner<V> {

        private final ModelProperty<? super M, V> prop;

        private ModelChangesValidationBuilderInner(ModelProperty<? super M, V> prop) {
            this.prop = prop;
        }

        public ModelChangesValidationBuilderInner<V> check(Constraint<V, Defect> constraint) {
            if (getResult().getValue().isPropChanged(prop)) {
                itemInternal(prop).check(constraint);
            }
            return this;
        }

        public ModelChangesValidationBuilderInner<V> weakCheck(Constraint<V, Defect> constraint) {
            if (getResult().getValue().isPropChanged(prop)) {
                itemInternal(prop).weakCheck(constraint);
            }
            return this;
        }

        public ModelChangesValidationBuilderInner<V> check(Constraint<V, Defect> constraint, When<V, Defect> when) {
            if (getResult().getValue().isPropChanged(prop)) {
                itemInternal(prop).check(constraint, when);
            }
            return this;
        }

        public ModelChangesValidationBuilderInner<V> checkBy(Validator<? extends V, Defect> validator) {
            if (getResult().getValue().isPropChanged(prop)) {
                itemInternal(prop).checkBy(validator);
            }
            return this;
        }

        public ModelChangesValidationBuilderInner<V> checkBy(Validator<? extends V, Defect> validator,
                                                             When<V, Defect> when) {
            if (getResult().getValue().isPropChanged(prop)) {
                itemInternal(prop).checkBy(validator, when);
            }
            return this;
        }

    }

}
