package ru.yandex.direct.libs.video;

import java.nio.charset.StandardCharsets;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.asynchttpclient.AsyncHttpClient;
import org.asynchttpclient.Request;
import org.asynchttpclient.RequestBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.asynchttp.AsyncHttpExecuteException;
import ru.yandex.direct.asynchttp.FetcherSettings;
import ru.yandex.direct.asynchttp.JsonParsableRequest;
import ru.yandex.direct.asynchttp.ParallelFetcher;
import ru.yandex.direct.asynchttp.ParallelFetcherFactory;
import ru.yandex.direct.asynchttp.ParsableRequest;
import ru.yandex.direct.libs.video.model.VideoBanner;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceChild;
import ru.yandex.direct.tracing.TraceProfile;

import static ru.yandex.direct.tracing.util.TraceUtil.X_YANDEX_TRACE;
import static ru.yandex.direct.tracing.util.TraceUtil.traceToHeader;
import static ru.yandex.direct.utils.net.YandexHttpHeaders.X_REQUEST_ID;
import static ru.yandex.direct.utils.net.YandexHttpHeaders.X_YANDEX_DIRECT_ADGROUP_ID;
import static ru.yandex.direct.utils.net.YandexHttpHeaders.X_YANDEX_DIRECT_CAMPAIGN_ID;
import static ru.yandex.direct.utils.net.YandexHttpHeaders.X_YANDEX_DIRECT_CLIENT_LOGIN;

/**
 * Клиент к ручке Яндекс.Видео, получает всю мета-информацию о видео
 */
@ParametersAreNonnullByDefault
public class VideoClient {
    private static final Logger logger = LoggerFactory.getLogger(VideoClient.class);

    public static final int CHUNK_SIZE = 5;

    // временный query-параметр до появления продовой ручки getMeta у Яндекс-Видео
    private static final String MAGIC_QUERY_NAME = "json_dump";
    private static final String RPS_LIMITED_QUOTA = "rps-limiter-quota";
    private static final String QUOTA_NAME = "video-prom";
    private static final String FILTER_PARAM = "family";
    private static final String FILTER_VALUE = "none";

    private final ParallelFetcherFactory fetcherFactory;
    private final VideoClientConfig config;

    public VideoClient(AsyncHttpClient asyncHttpClient, VideoClientConfig config) {
        this.config = config;
        this.fetcherFactory = new ParallelFetcherFactory(asyncHttpClient, new FetcherSettings()
                .withRequestRetries(config.getRequestRetries())
                .withRequestTimeout(config.getRequestTimeout())
                .withSoftTimeout(config.getSoftTimeout())
                .withParallel(config.getParallel()));
    }

    /**
     * Возвращает мета-информацию о видео из индекса Яндекс.Видео по списку их URL. Гарантирует возврат результатов в
     * том же порядке, в каком пришли URL.
     *
     * @param urls                 список URL видео
     * @param requestCorrelationId id запроса
     * @param clientLogin          логин клиента
     * @return мета-информация о видео
     */
    public List<VideoBanner> getMeta(List<String> urls,
                                     String requestCorrelationId,
                                     @Nullable String clientLogin) {
        return getMeta(urls, requestCorrelationId, null, null, clientLogin);
    }

    /**
     * Возвращает мета-информацию о видео из индекса Яндекс.Видео по списку их URL. Гарантирует возврат
     * результатов в
     * том же порядке, в каком пришли URL.
     *
     * @param urls                 список URL видео
     * @param requestCorrelationId id запроса
     * @param campaignId           id кампании
     * @param adGroupId            id группы объявлений
     * @param clientLogin          логин клиента
     * @return мета-информация о видео
     */
    public List<VideoBanner> getMeta(List<String> urls,
                                     String requestCorrelationId,
                                     @Nullable Long campaignId, @Nullable Long adGroupId,
                                     @Nullable String clientLogin) {
        try (TraceProfile profile = Trace.current().profile(config.getBaseUrl())) {
            TraceChild traceChild = Trace.current().child("Yandex.Video", "search");
            RequestBuilder builder = new RequestBuilder("GET")
                    .setUrl(config.getBaseUrl())
                    .addQueryParam("text", VideoClientUtils.buildUrlsQueryParam(urls))
                    .addQueryParam(MAGIC_QUERY_NAME, VideoBanner.JSON_PATH)
                    // специальный заголовок для увеличения квоты запросов на балансере
                    .addQueryParam(RPS_LIMITED_QUOTA, QUOTA_NAME)
                    // параметр для отключения умеренной фильтрации
                    .addQueryParam(FILTER_PARAM, FILTER_VALUE)
                    .setCharset(StandardCharsets.UTF_8)
                    // уникальный ID запроса, формируется на стороне клиента (Direct)
                    .addHeader(X_REQUEST_ID, requestCorrelationId)
                    .addHeader(X_YANDEX_DIRECT_CAMPAIGN_ID,
                            Optional.ofNullable(campaignId).map(String::valueOf).orElse(null))
                    .addHeader(X_YANDEX_DIRECT_ADGROUP_ID,
                            Optional.ofNullable(adGroupId).map(String::valueOf).orElse(null))
                    .addHeader(X_YANDEX_DIRECT_CLIENT_LOGIN, clientLogin)
                    .setHeader(X_YANDEX_TRACE, traceToHeader(traceChild));
            Request request = builder.build();

            logger.info("Request: {}", request);
            ParsableRequest<Object> parsableRequest = new JsonParsableRequest<>(
                    traceChild.getSpanId(),
                    request,
                    Object.class);
            try {
                ParallelFetcher<Object> parallelFetcher = fetcherFactory.getParallelFetcher();
                Map<String, VideoBanner> results = VideoBanner.parseBannerCollectionJson(
                        parallelFetcher.executeWithErrorsProcessing(parsableRequest).getSuccess());

                // По исходному url получаем url видеобаннера (он без протокола), а по нему - сам видеобаннер
                return StreamEx.of(urls)
                        .map(url -> StreamEx.of(results.keySet()).filter(url::endsWith).toList())
                        .map(keys -> keys.isEmpty() ? null : results.get(keys.get(0)))
                        .toList();
            } catch (AsyncHttpExecuteException ex) {
                return List.of();
            }
        }
    }
}
