package ru.yandex.direct.libs.video;

import java.net.URISyntaxException;
import java.util.List;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.apache.http.NameValuePair;
import org.apache.http.client.utils.URIBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.libs.video.model.VideoBanner;

@ParametersAreNonnullByDefault
public class VideoClientUtils {

    private static final Logger logger = LoggerFactory.getLogger(VideoClientUtils.class);

    private static final String YANDEX_HOST_PREFIX = "yandex";
    private static final String YANDEX_HOST_WWW_PREFIX = "www.yandex";
    private static final String EFIR_PATH = "/efir";
    private static final String STREAM_ID = "stream_id";

    /**
     * Возвращает URL-ы, скомпанованные для передачи в параметр запроса к Яндекс.Видео.
     * Для преобразования URL-ов используется {@link #buildUrlForQueryParam(String)}.
     * Все полученные значения конкатенируются через " | ".
     */
    static String buildUrlsQueryParam(List<String> urls) {
        return StreamEx.of(urls)
                .map(VideoClientUtils::buildUrlForQueryParam)
                .joining(" | ");
    }

    /**
     * Преобразует URL для передачи в параметр запроса к Яндекс.Видео.
     * Обычные URL-ы преобразует к виду "(url:{URL})", URL из Я.Эфира преобразует к виду "(uuid:{stream_id})".
     */
    static String buildUrlForQueryParam(String url) {
        try {
            URIBuilder parsedUrl = new URIBuilder(url);
            if (isEfirUrl(parsedUrl)) {
                String streamId = extractEfirStreamId(parsedUrl);
                if (streamId != null) {
                    return String.format("(uuid:%s)", streamId);
                }
            }
        } catch (URISyntaxException e) {
            logger.warn("Failed to create correct URL from String {}. Exception: {}", url, e);
        }
        return String.format("(url:%s)", url);
    }

    /**
     * Я.Видео для видео из Яндес.Эфира возвращает другой URL (и его нельзя изменить на стороне Видео), поэтому в этом
     * методе формируем правильный URL для видео из Эфира, а для других видео отдаём URL как есть.
     */
    public static String getUniformUrl(VideoBanner banner) {
        String url = banner.getVisibleUrl();
        if (url == null) {
            return banner.getUrl();
        }
        try {
            URIBuilder parsedUrl = new URIBuilder(url);
            if (isEfirUrl(parsedUrl)) {
                modifyEfirUrl(parsedUrl);
                return parsedUrl.toString();
            }
        } catch (URISyntaxException e) {
            logger.warn("Failed to create correct URL from String {}. Exception: {}", url, e);
        }
        return banner.getUrl();
    }

    public static boolean isEfirUrl(URIBuilder url) {
        String host = url.getHost().toLowerCase();
        return (host.startsWith(YANDEX_HOST_PREFIX) || host.startsWith(YANDEX_HOST_WWW_PREFIX))
                && url.getPath().equals(EFIR_PATH);
    }

    /**
     * Если передана валидная ссылка из Эфира, оставляет в параметрах запроса только переменную stream_id.
     * Иначе ничего не меняет.
     */
    public static void modifyEfirUrl(URIBuilder url) {
        String streamId = extractEfirStreamId(url);
        if (streamId != null) {
            url.setCustomQuery(String.format("%s=%s", STREAM_ID, streamId));
        }
    }

    @Nullable
    private static String extractEfirStreamId(URIBuilder url) {
        String streamId = url.getQueryParams().stream()
                .filter(p -> p.getName().equals(STREAM_ID))
                .map(NameValuePair::getValue)
                .findFirst()
                .orElse(null);
        if (streamId == null) {
            logger.error("Yandex Efir link without stream_id query parameter found: {}", url);
        }
        return streamId;
    }
}
