package ru.yandex.direct.xiva.client;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import javax.annotation.Nullable;

import com.google.common.collect.ImmutableMap;

import ru.yandex.direct.asynchttp.ParallelFetcherFactory;
import ru.yandex.direct.asynchttp.Result;
import ru.yandex.direct.http.smart.core.Smart;
import ru.yandex.direct.tvm.TvmIntegration;
import ru.yandex.direct.tvm.TvmService;
import ru.yandex.direct.xiva.client.model.AppInfo;
import ru.yandex.direct.xiva.client.model.BatchPush;
import ru.yandex.direct.xiva.client.model.Push;
import ru.yandex.direct.xiva.client.model.Recipient;
import ru.yandex.direct.xiva.client.model.SendBatchResponse;
import ru.yandex.direct.xiva.client.model.SendStatusSingleOrList;
import ru.yandex.direct.xiva.client.model.Signature;
import ru.yandex.direct.xiva.client.model.SubscribeResponse;
import ru.yandex.direct.xiva.client.model.Subscription;

import static ru.yandex.direct.http.smart.error.ErrorUtils.checkResultForErrors;


/**
 * Клиент для Xiva.
 *
 * @see <a href="https://console.push.yandex-team.ru/#api-reference-list-topic">Xiva API</a>
 * */
public class XivaClient {
    private static final String SUBSCRIPTION_TYPE_APP = "app";
    private static final String SUBSCRIPTION_TYPE_WEB = "webpush";
    private static final String SUBSCRIPTION_TYPE_SERVER = "url";
    private static final String PARAMETER_USER = "user";
    private static final String PARAMETER_SERVICE = "service";
    private static final String PARAMETER_CLIENT = "client";
    private static final String PARAMETER_SESSION = "session";
    private static final String PARAMETER_UUID = "uuid";
    private static final String PARAMETER_TOPIC = "topic";
    private static final String PARAMETER_TTL = "ttl";
    private static final String PARAMETER_EVENT = "event";
    private static final String PARAMETER_SUBSCRIPTION_ID = "subscription_id";
    private static final String PARAMETER_DEVICE = "device";
    private static final String PARAMETER_PLATFORM = "platform";
    private static final String PARAMETER_APP_NAME = "app_name";
    private static final String PARAMETER_CALLBACK = "callback";
    private static final String PARAMETER_FILTER = "filter";
    private static final String PARAMETER_EXTRA = "extra";
    private static final String PARAMETER_PUSH_TOKEN = "push_token";
    private static final String PARAMETER_SUBSCRIPTION = "subscription";

    private final XivaConfig config;
    private final Api api;

    private static final org.slf4j.Logger logger = org.slf4j.LoggerFactory.getLogger(XivaClient.class);

    public XivaClient(XivaConfig config, ParallelFetcherFactory parallelFetcherFactory,
                      TvmIntegration tvmIntegration, TvmService tvmService) {
        this.config = config;
        api = createApi(parallelFetcherFactory, tvmIntegration, tvmService);
    }

    private Api createApi(ParallelFetcherFactory parallelFetcherFactory,
                          TvmIntegration tvmIntegration, TvmService tvmService) {
        return Smart.builder()
                .withParallelFetcherFactory(parallelFetcherFactory)
                .withProfileName("xiva_client")
                .withBaseUrl(config.getServerHost())
                .useTvm(tvmIntegration, tvmService)
                .build()
                .create(Api.class);
    }


    /**
     * Получить одноразовый секретный ключ для подписки по протоколу WebSocket.
     * Если запрос к xiva возвращает ошибку, считаем, что получить подпись не удалось и возвращаем null.
     *
     * @param user идентификатор пользователя.
     *
     * @return подпись или null, если подпись получить не удалось.
     * */
    @Nullable
    public Signature getSignature(String user) {
        Result<Signature> result = api.getSignature(
                ImmutableMap.of(PARAMETER_SERVICE, config.getServiceName(), PARAMETER_USER, user)
        ).execute();
        try {
            checkResultForErrors(result, XivaClientException::new);
        } catch (Exception e) {
            logger.error("Error while xiva secret_sign request", e);
            return null;
        }
        return result.getSuccess();
    }

    @Nullable
    private String subscribeMobileApp(AppInfo app, String token,
                                      @Nullable String filter, @Nullable String extra, @Nullable String topic) {
        String body = PARAMETER_PUSH_TOKEN +  "=" + token;
        if (!Objects.isNull(filter)) {
            body = body + "&" + PARAMETER_FILTER + "=" + filter;
        }
        if (!Objects.isNull(extra)) {
            body = body + "&" + PARAMETER_EXTRA + "=" + extra;
        }

        HashMap<String, String> params = new HashMap<>(Map.of(
                PARAMETER_SERVICE, config.getServiceName(),
                PARAMETER_USER, app.getUser(),
                PARAMETER_CLIENT, app.getClient(),
                PARAMETER_UUID, app.getUuid(),
                PARAMETER_DEVICE, app.getDevice(),
                PARAMETER_PLATFORM, app.getPlatform(),
                PARAMETER_APP_NAME, app.getAppName()
        ));
        if (!Objects.isNull(topic)) {
            params.put(PARAMETER_TOPIC, topic);
        }

        Result<SubscribeResponse> result = api.subscribe(
                SUBSCRIPTION_TYPE_APP,
                params,
                body
        ).execute();
        try {
            checkResultForErrors(result, XivaClientException::new);
        } catch (Exception e) {
            logger.error("Error while xiva subscribe request", e);
            return null;
        }
        return result.getSuccess().getId();
    }

    /**
     * Подписать мобильное приложение на персональные уведомления.
     *
     * @param app информация о приложении.
     * @param token в зависимости от платформы: device token (apns), registration id (fcm) или channel URI (mpns, wns).
     * @param filter правила фильтрации.
     * @param extra дополнительная информация.
     *
     * @return id подписки или null, если не смогли подписаться.
     * */
    @Nullable
    public String subscribeMobileApp(AppInfo app, String token, @Nullable String filter, @Nullable String extra) {
        return subscribeMobileApp(app, token, filter, extra, null);
    }

    /**
     * Подписать мобильное приложение на топик.
     *
     * @param topic название топика.
     * @param app информация о приложении.
     * @param token в зависимости от платформы: device token (apns), registration id (fcm) или channel URI (mpns, wns).
     * @param filter правила фильтрации.
     * @param extra дополнительная информация.
     *
     * @return id подписки или null, если не смогли подписаться.
     * */
    @Nullable
    public String subscribeMobileAppToTopic(String topic, AppInfo app, String token,
                                            @Nullable String filter, @Nullable String extra) {
        return subscribeMobileApp(app, token, filter, extra, topic);
    }

    /**
     * Отписать мобильное приложение от персональных уведомлений.
     *
     * @param user идентификатор пользователя.
     * @param uuid уникальный идентификатор инсталляции приложения.
     * */
    public void unsubscribeMobileApp(String user, String uuid) {
        Result<String> result = api.unsubscribe(
                SUBSCRIPTION_TYPE_APP,
                ImmutableMap.of(
                        PARAMETER_SERVICE, config.getServiceName(),
                        PARAMETER_USER, user,
                        PARAMETER_UUID, uuid
                )
        ).execute();
        try {
            checkResultForErrors(result, XivaClientException::new);
        } catch (Exception e) {
            logger.error("Error while xiva unsubscribe request", e);
        }
    }

    /**
     * Отписать мобильное приложение от топика.
     *
     * @param topic название топика.
     * @param uuid уникальный идентификатор инсталляции приложения.
     * */
    public void unsubscribeMobileFromTopic(String topic, String uuid) {
        Result<String> result = api.unsubscribe(
                SUBSCRIPTION_TYPE_APP,
                ImmutableMap.of(
                        PARAMETER_SERVICE, config.getServiceName(),
                        PARAMETER_TOPIC, topic,
                        PARAMETER_UUID, uuid
                )
        ).execute();
        try {
            checkResultForErrors(result, XivaClientException::new);
        } catch (Exception e) {
            logger.error("Error while xiva unsubscribe request", e);
        }
    }

    @Nullable
    private String subscribeWeb(String user, String client, String session, String subscription,
                                @Nullable String filter, @Nullable String extra, @Nullable String topic) {
        String body = PARAMETER_SUBSCRIPTION + "=" + subscription;
        if (!Objects.isNull(filter)) {
            body = body + "&" + PARAMETER_FILTER + "=" + filter;
        }
        if (!Objects.isNull(extra)) {
            body = body + "&" + PARAMETER_EXTRA + "=" + extra;
        }

        HashMap<String, String> params = new HashMap<>(Map.of(
                PARAMETER_SERVICE, config.getServiceName(),
                PARAMETER_USER, user,
                PARAMETER_CLIENT, client,
                PARAMETER_SESSION, session
        ));
        if (!Objects.isNull(topic)) {
            params.put(PARAMETER_TOPIC, topic);
        }

        Result<SubscribeResponse> result = api.subscribe(
                SUBSCRIPTION_TYPE_WEB,
                params,
                body
        ).execute();
        try {
            checkResultForErrors(result, XivaClientException::new);
        } catch (Exception e) {
            logger.error("Error while xiva subscribe request", e);
            return null;
        }
        return result.getSuccess().getId();
    }

    /**
     * Подписать браузер на персональные уведомления.
     *
     * @param user идентификатор пользователя.
     * @param client название семейства клиентов.
     * @param session идентификатор экземпляра клиента.
     * @param subscription подписка Web Push в формате JSON, которую выдает push manager в браузере.
     * @param filter правила фильтрации.
     * @param extra дополнительная информация.
     *
     * @return id подписки или null, если подписаться не удалось.
     * */
    @Nullable
    public String subscribeWeb(String user, String client, String session, String subscription,
                               @Nullable String filter, @Nullable String extra) {
        return subscribeWeb(user, client, session, subscription, filter, extra, null);
    }

    /**
     * Подписать браузер на топик.
     *
     * @param topic название топика.
     * @param user идентификатор пользователя.
     * @param client название семейства клиентов.
     * @param session идентификатор экземпляра клиента.
     * @param subscription подписка Web Push в формате JSON, которую выдает push manager в браузере.
     * @param filter правила фильтрации.
     * @param extra дополнительная информация.
     *
     * @return id подписки или null, если подписаться не удалось.
     * */
    @Nullable
    public String subscribeWebToTopic(
            String topic, String user, String client, String session, String subscription,
            @Nullable String filter, @Nullable String extra) {
        return subscribeWeb(user, client, session, subscription, filter, extra, topic);
    }

    /**
     * Отписать браузер от персональных уведомлений.
     *
     * @param user идентификатор пользователя.
     * @param session идентификатор экземпляра клиента.
     * */
    public void unsubscribeWeb(String user, String session) {
        Result<String> result = api.unsubscribe(
                SUBSCRIPTION_TYPE_WEB,
                ImmutableMap.of(
                        PARAMETER_SERVICE, config.getServiceName(),
                        PARAMETER_USER, user,
                        PARAMETER_SESSION, session
                )
        ).execute();
        try {
            checkResultForErrors(result, XivaClientException::new);
        } catch (Exception e) {
            logger.error("Error while xiva unsubscribe request", e);
        }
    }

    /**
     * Отписать браузер от топика.
     *
     * @param topic название топика.
     * @param session идентификатор экземпляра клиента.
     * */
    public void unsubscribeWebFromTopic(String topic, String session) {
        Result<String> result = api.unsubscribe(
                SUBSCRIPTION_TYPE_WEB,
                ImmutableMap.of(
                        PARAMETER_SERVICE, config.getServiceName(),
                        PARAMETER_TOPIC, topic,
                        PARAMETER_SESSION, session
                )
        ).execute();
        try {
            checkResultForErrors(result, XivaClientException::new);
        } catch (Exception e) {
            logger.error("Error while xiva unsubscribe request", e);
        }
    }

    @Nullable
    private String subscribeServer(String user, String client, String session, String callback,
                                   @Nullable String filter, @Nullable String topic) {
        String body = "";
        if (!Objects.isNull(filter)) {
            body = PARAMETER_FILTER + "=" + filter;
        }

        HashMap<String, String> params = new HashMap<>(Map.of(
                PARAMETER_SERVICE, config.getServiceName(),
                PARAMETER_USER, user,
                PARAMETER_CLIENT, client,
                PARAMETER_SESSION, session,
                PARAMETER_CALLBACK, callback
        ));
        if (!Objects.isNull(topic)) {
            params.put(PARAMETER_TOPIC, topic);
        }

        Result<SubscribeResponse> result = api.subscribe(
                SUBSCRIPTION_TYPE_SERVER,
                params,
                body
        ).execute();
        try {
            checkResultForErrors(result, XivaClientException::new);
        } catch (Exception e) {
            logger.error("Error while xiva subscribe request", e);
            return null;
        }
        return result.getSuccess().getId();
    }

    /**
     * Подписать сервер на персональные уведомления через HTTP вызовы.
     *
     * @param user идентификатор пользователя.
     * @param client название семейства клиентов.
     * @param session идентификатор экземпляра клиента.
     * @param callbackURL HTTP URL для обратного вызова.
     * @param filter правила фильтрации.
     *
     * @return id подписки или null, если подписаться не удалось.
     * */
    @Nullable
    public String subscribeServer(String user, String client, String session, String callbackURL,
                                  @Nullable String filter) {
        return subscribeServer(user, client, session, callbackURL, filter, null);
    }

    /**
     * Подписать сервер на топик (HTTP вызовы).
     *
     * @param topic название топика.
     * @param user идентификатор пользователя.
     * @param client название семейства клиентов.
     * @param session идентификатор экземпляра клиента.
     * @param callbackURL HTTP URL для обратного вызова.
     * @param filter правила фильтрации.
     *
     * @return id подписки или null, если подписаться не удалось.
     * */
    @Nullable
    public String subscribeServerToTopic(
            String topic, String user, String client, String session, String callbackURL, String filter) {
        return subscribeServer(user, client, session, callbackURL, filter, topic);
    }

    /**
     * Отписать сервер от персональных уведомлений через HTTP вызовы.
     *
     * @param user идентификатор пользователя.
     * @param subscription идентификатор подписки.
     * */
    public void unsubscribeServer(String user, String subscription) {
        Result<String> result = api.unsubscribe(
                SUBSCRIPTION_TYPE_SERVER,
                ImmutableMap.of(
                        PARAMETER_SERVICE, config.getServiceName(),
                        PARAMETER_USER, user,
                        PARAMETER_SUBSCRIPTION_ID, subscription
                )
        ).execute();
        try {
            checkResultForErrors(result, XivaClientException::new);
        } catch (Exception e) {
            logger.error("Error while xiva unsubscribe request", e);
        }
    }

    /**
     * Отписать сервер от топика (HTTP вызовы).
     *
     * @param topic название топика.
     * @param user идентификатор пользователя.
     * @param subscription идентификатор подписки.
     * */
    public void unsubscribeServerFromTopic(String topic, String user, String subscription) {
        Result<String> result = api.unsubscribe(
                SUBSCRIPTION_TYPE_SERVER,
                ImmutableMap.of(
                        PARAMETER_SERVICE, config.getServiceName(),
                        PARAMETER_USER, user,
                        PARAMETER_TOPIC, topic,
                        PARAMETER_SUBSCRIPTION_ID, subscription
                )
        ).execute();
        try {
            checkResultForErrors(result, XivaClientException::new);
        } catch (Exception e) {
            logger.error("Error while xiva unsubscribe request", e);
        }
    }


    /**
     * Отправить уведомление пользователю.
     *
     * @param user идентификатор пользователя.
     * @param push уведомление для отправки.
     * @param ttl время в секундах, в течение которого уведомление актуально.
     * */
    public void send(String user, Push push, @Nullable Integer ttl) {
        HashMap<String, String> params = new HashMap<>(Map.of(
                PARAMETER_USER, user,
                PARAMETER_EVENT, push.getDescription(),
                PARAMETER_SERVICE, config.getServiceName()
        ));
        if (!Objects.isNull(ttl)) {
            params.put(PARAMETER_TTL, String.valueOf(ttl));
        }
        Result<String> result = api.send(
                params,
                push
        ).execute();
        try {
            checkResultForErrors(result, XivaClientException::new);
        } catch (Exception e) {
            logger.error("Error while xiva send request", e);
        }
    }

    /**
     * Отправить сообщение группе пользователей.
     *
     * @param recipients список получаетелей (пользователей или отдельных подписок).
     * @param push уведомление для отправки.
     * @param ttl время в секундах, в течение которого уведомление актуально.
     *
     * @return статус отправки для каждого получателя или null, если возникла ошибка при обращении к серверу.
     * */
    public List<SendStatusSingleOrList> sendBatch(List<Recipient> recipients, Push push, @Nullable Integer ttl) {
        HashMap<String, String> params = new HashMap<>(Map.of(
                PARAMETER_EVENT, push.getDescription(),
                PARAMETER_SERVICE, config.getServiceName()
        ));
        if (!Objects.isNull(ttl)) {
            params.put(PARAMETER_TTL, String.valueOf(ttl));
        }
        Result<SendBatchResponse> result = api.sendBatch(
                params,
                new BatchPush(push, recipients)
        ).execute();
        try {
            checkResultForErrors(result, XivaClientException::new);
        } catch (Exception e) {
            logger.error("Error while xiva batch_send request", e);
            return null;
        }
        return result.getSuccess().getStatuses();
    }

    /**
     * Получить список подписок пользователя.
     *
     * @param user идентификатор пользователя.
     *
     * @return список подписок или null, если не удалось его получить.
     * */
    public List<Subscription> subscriptions(String user) {
        Result<List<Subscription>> result = api.getSubscriptions(
                ImmutableMap.of(PARAMETER_SERVICE, config.getServiceName(), PARAMETER_USER, user)
        ).execute();
        try {
            checkResultForErrors(result, XivaClientException::new);
        } catch (Exception e) {
            logger.error("Error while xiva list request", e);
            return null;
        }
        return result.getSuccess();
    }

    /**
     * Отправить уведомление в топик.
     *
     * @param topic название топика.
     * @param push уведомление для отправки.
     * @param ttl время в секундах, в течение которого уведомление актуально.
     * */
    public void sendToTopic(String topic, Push push, @Nullable Integer ttl) {
        HashMap<String, String> params = new HashMap<>(Map.of(
                PARAMETER_TOPIC, topic,
                PARAMETER_EVENT, push.getDescription(),
                PARAMETER_SERVICE, config.getServiceName()
        ));
        if (!Objects.isNull(ttl)) {
            params.put(PARAMETER_TTL, String.valueOf(ttl));
        }
        Result<String> result = api.send(
                params,
                push
        ).execute();
        try {
            checkResultForErrors(result, XivaClientException::new);
        } catch (Exception e) {
            logger.error("Error while xiva send request", e);
        }
    }

    /**
     * Получить список подписок на топик.
     *
     * @param topic название топика.
     *
     * @return список подписок или null, если не удалось его получить.
     * */
    public List<Subscription> subscriptionsOnTopics(String topic) {
        Result<List<Subscription>> result = api.getSubscriptions(
                ImmutableMap.of(PARAMETER_SERVICE, config.getServiceName(), PARAMETER_TOPIC, topic)
        ).execute();
        try {
            checkResultForErrors(result, XivaClientException::new);
        } catch (Exception e) {
            logger.error("Error while xiva list request", e);
            return null;
        }
        return result.getSuccess();
    }
}
