package ru.yandex.direct.audience.client;

import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;

import org.asynchttpclient.AsyncHttpClient;
import org.asynchttpclient.request.body.multipart.ByteArrayPart;

import ru.yandex.direct.asynchttp.FetcherSettings;
import ru.yandex.direct.asynchttp.ParallelFetcherFactory;
import ru.yandex.direct.asynchttp.Result;
import ru.yandex.direct.audience.client.exception.YaAudienceClientException;
import ru.yandex.direct.audience.client.model.AudienceSegment;
import ru.yandex.direct.audience.client.model.ConfirmSegmentRequest;
import ru.yandex.direct.audience.client.model.ConfirmingSegment;
import ru.yandex.direct.audience.client.model.CreateExperimentRequest;
import ru.yandex.direct.audience.client.model.CreateExperimentRequestEnvelope;
import ru.yandex.direct.audience.client.model.CreateExperimentResponseEnvelope;
import ru.yandex.direct.audience.client.model.DeleteSegmentResponse;
import ru.yandex.direct.audience.client.model.SegmentContentType;
import ru.yandex.direct.audience.client.model.SegmentResponse;
import ru.yandex.direct.audience.client.model.SegmentsResponse;
import ru.yandex.direct.audience.client.model.SetExperimentGrantRequest;
import ru.yandex.direct.audience.client.model.SetExperimentGrantRequestEnvelope;
import ru.yandex.direct.audience.client.model.SetExperimentGrantResponseEnvelope;
import ru.yandex.direct.audience.client.model.geosegment.YaAudienceCreateGeoSegmentRequest;
import ru.yandex.direct.audience.client.model.geosegment.YaAudienceGeoPoint;
import ru.yandex.direct.audience.client.model.geosegment.YaAudienceGeoSegment;
import ru.yandex.direct.audience.client.model.geosegment.YaAudienceGeoSegmentType;
import ru.yandex.direct.http.smart.core.Call;
import ru.yandex.direct.http.smart.core.Smart;
import ru.yandex.direct.liveresource.LiveResource;
import ru.yandex.direct.liveresource.MemoryLiveResource;

import static ru.yandex.direct.audience.client.YaAudienceClientUtils.getException;
import static ru.yandex.direct.http.smart.error.ErrorUtils.checkResultForErrors;

public class YaAudienceClient {
    private static final Duration REQUEST_TIMEOUT = Duration.ofSeconds(6);
    private static final Duration UPLOAD_REQUEST_TIMEOUT = Duration.ofSeconds(40);
    private static final String ADDITION_MODIFICATION_TYPE_NAME = "addition";

    private final Api api;
    private final UploadApi uploadApi;
    private final LiveResource token;

    public YaAudienceClient(AsyncHttpClient asyncHttpClient, LiveResource token, String baseAudienceApiUrl) {
        this.api = createAudienceApi(Api.class, REQUEST_TIMEOUT, asyncHttpClient, baseAudienceApiUrl);
        this.uploadApi = createAudienceApi(UploadApi.class, UPLOAD_REQUEST_TIMEOUT, asyncHttpClient,
                baseAudienceApiUrl);
        this.token = token;
    }

    //for tests only
    public YaAudienceClient(Api api, UploadApi uploadApi) {
        this.api = api;
        this.uploadApi = uploadApi;
        this.token = new MemoryLiveResource("");
    }

    private <T> T createAudienceApi(Class<T> apiClass, Duration timeout, AsyncHttpClient asyncHttpClient,
                                    String baseAudienceApiUrl) {
        return Smart.builder()
                .withParallelFetcherFactory(new ParallelFetcherFactory(asyncHttpClient, new FetcherSettings()
                        .withRequestTimeout(timeout)))
                .addHeaderConfigurator(h -> h.add("Authorization", "OAuth " + token.getContent()))
                .withProfileName("ya_audience_client")
                .withBaseUrl(baseAudienceApiUrl)
                .build()
                .create(apiClass);
    }

    /**
     * Создает геосегмент в Яндекс.Аудиториях.
     *
     * @param ulogin         логин пользователя, для которого создается сегмент
     * @param segmentName    название сегмента
     * @param radius         радиус окружностей
     * @param points         центры окружностей
     * @param geoSegmentType тип координат для создания сегмента
     * @param periodLength   период посещений указанных мест (для условия 'посетил указанные места N раз за период')
     * @param timesQuantity  частота посещений указанных мест (для условия 'посетил указанные места N раз за период')
     * @return информация о созданном сегменте
     */
    public SegmentResponse createGeoSegment(
            String ulogin,
            String segmentName,
            int radius,
            Set<YaAudienceGeoPoint> points,
            YaAudienceGeoSegmentType geoSegmentType,
            Integer periodLength,
            Integer timesQuantity) {
        var request = new YaAudienceCreateGeoSegmentRequest(
                new YaAudienceGeoSegment(segmentName, radius, points, geoSegmentType, periodLength, timesQuantity));

        return executeCall(api.createGeoSegment(ulogin, request));
    }

    public List<AudienceSegment> getSegments(String ulogin) {
        SegmentsResponse segmentsResponse = executeCall(api.getSegments(ulogin));
        return segmentsResponse.getAudienceSegments();
    }

    public AudienceSegment confirmSegment(String ulogin, Long segmentId, String segmentName,
                                          SegmentContentType contentType) {
        return confirmSegment(ulogin, segmentId, segmentName, contentType, null);
    }

    public AudienceSegment confirmYuidSegment(String ulogin, Long segmentId, String segmentName) {
        return confirmYuidSegment(ulogin, segmentId, segmentName, null);
    }

    public AudienceSegment confirmYuidSegment(String ulogin, Long segmentId, String segmentName,
                                              @Nullable String lang) {
        return confirmSegment(ulogin, segmentId, segmentName, SegmentContentType.YUID, lang);
    }

    private AudienceSegment confirmSegment(String ulogin, Long segmentId, String segmentName,
                                           SegmentContentType contentType, @Nullable String lang) {
        return executeSegmentCall(api.confirmSegment(segmentId, ulogin, lang,
                new ConfirmSegmentRequest(new ConfirmingSegment()
                        .withSegmentId(segmentId)
                        .withName(segmentName)
                        .withHashed(false)
                        .withContentType(contentType))));
    }

    public AudienceSegment uploadSegment(String ulogin, byte[] content) {
        return executeSegmentCall(uploadApi.uploadFile(ulogin, new ByteArrayPart("file", content,
                "application/octet-stream", StandardCharsets.UTF_8, "data.tsv")));
    }

    public AudienceSegment modifySegment(String ulogin, Long segmentId, byte[] content) {
        return executeSegmentCall(uploadApi.modifySegment(segmentId, ulogin,
                ADDITION_MODIFICATION_TYPE_NAME,
                new ByteArrayPart("file", content, "application/octet-stream", StandardCharsets.UTF_8,
                        "data.tsv")));
    }

    /**
     * Удаляет геосегмент в Яндекс.Аудиториях.
     *
     * @param segmentId - id сегмента, который нужно удалить.
     * @return успешно ли был удален сегмент
     */
    public boolean deleteSegment(Long segmentId) {
        DeleteSegmentResponse segmentResponse = executeCall(api.deleteSegment(segmentId));
        return segmentResponse.getSuccess();
    }

    public CreateExperimentResponseEnvelope createExperiment(String ulogin,
                                                             CreateExperimentRequest createExperimentRequest) {
        Result<CreateExperimentResponseEnvelope> result = api.createExpertiment(ulogin,
                new CreateExperimentRequestEnvelope(createExperimentRequest)).execute();
        checkResultForErrors(result, YaAudienceClientException::new);
        return result.getSuccess();
    }

    public CreateExperimentResponseEnvelope updateExperiment(String ulogin,
                                                             Long experimentId,
                                                             CreateExperimentRequest createExperimentRequest) {
        Result<CreateExperimentResponseEnvelope> result = api.updateExperiment(experimentId, ulogin,
                new CreateExperimentRequestEnvelope(createExperimentRequest)).execute();
        checkResultForErrors(result, YaAudienceClientException::new);
        return result.getSuccess();
    }

    public SetExperimentGrantResponseEnvelope setExperimentGrant(Long experimentId,
                                                                 SetExperimentGrantRequest setExperimentGrantRequest) {
        Result<SetExperimentGrantResponseEnvelope> result = api.setExperimentGrant(experimentId,
                new SetExperimentGrantRequestEnvelope(setExperimentGrantRequest)).execute();

        checkResultForErrors(result, YaAudienceClientException::new);
        return result.getSuccess();
    }

    private AudienceSegment executeSegmentCall(Call<SegmentResponse> call) {
        SegmentResponse segmentResponse = executeCall(call);
        return segmentResponse.getAudienceSegment();
    }

    private <T> T executeCall(Call<T> call) {
        Result<T> result = call.execute();
        checkResultForErrorsWithSegmentCheck(result);
        return result.getSuccess();
    }

    private <T> void checkResultForErrorsWithSegmentCheck(Result<T> result) {
        if (result.getErrors() != null) {
            String message = "Exception while dealing with ya audience client";
            YaAudienceClientException exception = getException(result.getErrors(), message);
            result.getErrors().forEach(exception::addSuppressed);
            throw exception;
        }
    }
}
