package ru.yandex.direct.audience.client;

import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.annotation.Nonnull;

import org.asynchttpclient.Response;

import ru.yandex.direct.asynchttp.ErrorResponseWrapperException;
import ru.yandex.direct.audience.client.exception.SegmentNotModifiedException;
import ru.yandex.direct.audience.client.exception.YaAudienceClientException;
import ru.yandex.direct.audience.client.exception.YaAudienceClientTypedException;
import ru.yandex.direct.audience.client.model.AudienceErrorResponse;
import ru.yandex.direct.utils.JsonUtils;

class YaAudienceClientUtils {
    private static final String SEGMENT_NOT_CHANGED_MESSAGE = "Данные сегмента не изменились";
    private static final String ERROR_TYPE = "error_type";

    private YaAudienceClientUtils() {
    }

    static boolean errorIsSegmentNotChanged(@Nonnull Throwable throwable) {
        if (!(throwable instanceof ErrorResponseWrapperException)) {
            return false;
        }
        ErrorResponseWrapperException error = (ErrorResponseWrapperException) throwable;
        return Optional.of(error)
                .map(ErrorResponseWrapperException::getResponse)
                .filter(Objects::nonNull)
                .map(Response::getResponseBody)
                .map(responseBody -> JsonUtils.fromJson(responseBody, AudienceErrorResponse.class))
                .map(AudienceErrorResponse::getMessage)
                .map(message -> message.equals(SEGMENT_NOT_CHANGED_MESSAGE))
                .orElse(false);
    }

    static YaAudienceClientException getException(@Nonnull List<Throwable> errors, @Nonnull String message) {
        if (errors.size() == 1) {
            if (errorIsSegmentNotChanged(errors.get(0))) {
                return new SegmentNotModifiedException(message);
            } else if (errorIsTyped(errors.get(0))) {
                return toYaAudienceTypedClientException(errors.get(0));
            }
        }

        return new YaAudienceClientException(message);
    }

    private static YaAudienceClientTypedException toYaAudienceTypedClientException(@Nonnull Throwable throwable) {
        String errorType = obtainErrorInfoByKey(ERROR_TYPE, throwable);
        String message = obtainErrorInfoByKey("message", throwable);
        return new YaAudienceClientTypedException(message, errorType);
    }

    private static boolean errorIsTyped(@Nonnull Throwable throwable) {
        return obtainErrorInfoByKey(ERROR_TYPE, throwable) != null;
    }

    private static String obtainErrorInfoByKey(@Nonnull String key, @Nonnull Throwable throwable) {
        if (throwable.getCause() != null && throwable.getCause().getMessage() != null) {
            Pattern pattern = Pattern.compile("\"" + key + "\":\"(?<keyText>(\\w|\\s|[А-Яа-я]|[.,!?\\-])+)\"");
            Matcher matcher = pattern.matcher(throwable.getCause().getMessage());
            if (matcher.find()) {
                return matcher.group("keyText");
            }
        }
        return null;
    }
}
