package ru.yandex.direct.captcha.passport;

import java.io.IOException;
import java.io.StringReader;
import java.time.Duration;
import java.util.List;
import java.util.Objects;

import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Unmarshaller;

import com.google.common.base.Preconditions;
import com.google.common.net.HttpHeaders;
import org.asynchttpclient.AsyncHttpClient;
import org.asynchttpclient.Param;
import org.asynchttpclient.RequestBuilder;

import ru.yandex.direct.asynchttp.FetcherSettings;
import ru.yandex.direct.asynchttp.ParallelFetcher;
import ru.yandex.direct.asynchttp.ParallelFetcherFactory;
import ru.yandex.direct.asynchttp.ParsableStringRequest;
import ru.yandex.direct.asynchttp.Result;
import ru.yandex.direct.captcha.passport.entity.CaptchaType;
import ru.yandex.direct.captcha.passport.entity.check.request.CaptchaChecksParameter;
import ru.yandex.direct.captcha.passport.entity.check.request.CaptchaHttpsMode;
import ru.yandex.direct.captcha.passport.entity.check.response.CaptchaCheckResponse;
import ru.yandex.direct.captcha.passport.entity.generate.response.CaptchaGenerateResponse;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;

import static java.util.Arrays.asList;
import static java.util.stream.Collectors.toList;

/**
 * Клиент к Yandex.Captcha.
 *
 * @see <a href="https://doc.yandex-team.ru/Passport/captcha/concepts/about.xml">Yandex.Captcha</a>
 */
public class YandexCaptchaClient {
    private static final Duration REQUEST_TIMEOUT = Duration.ofSeconds(2);
    private static final int REQUEST_RETRIES = 2;
    private static final Duration SOFT_TIMEOUT = Duration.ofMillis(200);

    private final ParallelFetcherFactory parallelFetcherFactory;
    private final JAXBContext generateContext;
    private final JAXBContext checkContext;
    private final YandexCaptchaConfig captchaConfig;

    public YandexCaptchaClient(YandexCaptchaConfig captchaConfig, AsyncHttpClient asyncHttpClient) {
        this.captchaConfig = captchaConfig;

        FetcherSettings settings = new FetcherSettings()
                .withRequestTimeout(REQUEST_TIMEOUT)
                .withRequestRetries(REQUEST_RETRIES)
                .withSoftTimeout(SOFT_TIMEOUT);
        parallelFetcherFactory = new ParallelFetcherFactory(asyncHttpClient, settings);

        try {
            this.generateContext = JAXBContext.newInstance(CaptchaGenerateResponse.class);
            this.checkContext = JAXBContext.newInstance(CaptchaCheckResponse.class);
        } catch (JAXBException ex) {
            throw new YandexCaptchaException(ex);
        }
    }

    private <T> T doRequest(CaptchaMethod method, Unmarshaller unmarshaller, List<Param> requestParameters)
            throws IOException, JAXBException {
        RequestBuilder builder = new RequestBuilder()
                .setUrl(captchaConfig.getUrlBase() + "/" + method.path)
                .addQueryParams(requestParameters.stream().filter(Objects::nonNull).collect(toList()))
                .addHeader(HttpHeaders.ACCEPT, "application/xml")
                .addHeader(HttpHeaders.ACCEPT_CHARSET, "UTF-8")
                .addHeader(HttpHeaders.HOST, captchaConfig.getHttpHostHeader());

        try (
                TraceProfile ignore = Trace.current().profile("captcha", method.path);
                ParallelFetcher<String> fetcher = parallelFetcherFactory.getParallelFetcher();) {
            Result<String> result;
            try {
                result = fetcher.execute(new ParsableStringRequest(0, builder.build()));
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new YandexCaptchaException("Captcha server exception", e);
            }

            if (result.getErrors() != null && !result.getErrors().isEmpty()) {
                throw new YandexCaptchaException("Captcha server error", result.getErrors());
            }

            @SuppressWarnings("unchecked")
            T ret = (T) unmarshaller.unmarshal(new StringReader(result.getSuccess()));
            return ret;
        }
    }

    /**
     * Аналогичен вызову {@code check(null, captchaKey, userInput) }
     */
    public CaptchaCheckResponse check(String captchaKey, String userInput) {
        return check(null, captchaKey, userInput);
    }

    /**
     * Проверяет распознанное значение капчи на корректность.
     *
     * @param captchaType тип капчи. Должен совпадать с типом, указанным при генерации капчи; может быть null
     * @param captchaKey  ключ капчи
     * @param userInput   значение, которое ввел(распознал) пользователь.
     */
    public CaptchaCheckResponse check(CaptchaType captchaType, String captchaKey, String userInput) {
        Preconditions.checkNotNull(captchaKey);
        Preconditions.checkNotNull(userInput);
        try {
            CaptchaCheckResponse response = doRequest(CaptchaMethod.CHECK, checkContext.createUnmarshaller(),
                    asList(captchaType == null ? null : captchaType.asRequestParameter(),
                            new Param("rep", userInput),
                            new Param("key", captchaKey))
            );

            if (response.getCheckStatus() == null) {
                throw new YandexCaptchaException("check status is null after xml unmarshalling");
            }
            return response;
        } catch (JAXBException | IOException ex) {
            throw new YandexCaptchaException(ex);
        }
    }

    /**
     * Запрашивает Captcha-сервер подготовить Captcha-тест.
     *
     * @param captchaType  тип капчи. Может быть null
     * @param attemptCount максимальное число попыток распознавания капчи
     * @param httpsMode    надо ли использовать https режим?; может быть null
     */
    public CaptchaGenerateResponse generate(CaptchaType captchaType, int attemptCount, CaptchaHttpsMode httpsMode) {
        try {
            CaptchaGenerateResponse response = doRequest(CaptchaMethod.GENERATE, generateContext.createUnmarshaller(),
                    asList(captchaType == null ? null : captchaType.asRequestParameter(),
                            httpsMode == null ? null : httpsMode.asRequestParameter(),
                            CaptchaChecksParameter.asRequestParameter(attemptCount))
            );

            if (response.getUrl() == null || response.getRequestId() == null) {
                throw new YandexCaptchaException("url or requestId is null after xml unmarshalling");
            }
            return response;
        } catch (JAXBException | IOException ex) {
            throw new YandexCaptchaException(ex);
        }
    }

    /**
     * Аналогичен вызову {@code generate(captchaType, attemptCount, null) }
     */
    public CaptchaGenerateResponse generate(CaptchaType captchaType, int attemptCount) {
        return generate(captchaType, attemptCount, null);
    }

    /**
     * Аналогичен вызову {@code generate(null, attemptCount, null) }
     */
    public CaptchaGenerateResponse generate(int attemptCount) {
        return generate(null, attemptCount, null);
    }

    enum CaptchaMethod {
        GENERATE("generate"),
        CHECK("check");

        String path;

        CaptchaMethod(String path) {
            this.path = path;
        }
    }
}
