package ru.yandex.direct.sender;

import java.time.Duration;
import java.util.Collection;
import java.util.Map;
import java.util.function.Predicate;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.asynchttpclient.AsyncHttpClient;
import org.asynchttpclient.Param;
import org.asynchttpclient.Realm;
import org.asynchttpclient.RequestBuilder;

import ru.yandex.direct.asynchttp.FetcherSettings;
import ru.yandex.direct.asynchttp.ParallelFetcher;
import ru.yandex.direct.asynchttp.ParallelFetcherFactory;
import ru.yandex.direct.asynchttp.Result;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;

import static com.google.common.base.Preconditions.checkNotNull;
import static io.netty.handler.codec.http.HttpHeaderNames.CONTENT_TYPE;
import static io.netty.handler.codec.http.HttpHeaderValues.APPLICATION_X_WWW_FORM_URLENCODED;
import static java.util.Collections.singletonList;
import static org.asynchttpclient.util.HttpConstants.Methods.POST;

/**
 * Клиент к Yandex Sender
 * <p>
 * Про таймаут: при использовании синхронного режима отправки - процесс может занимать 10-15 секунд,
 * при асинхронной - 1-5 секунд. Текущее значение {@link #REQUEST_TIMEOUT} плохо покрывает оба случая,
 * поэтому если понадобится синхронная отправка - стоит сделать таймаут конфигурируемым на клиент (через конфиг)
 * или на запрос (через отдельный метод отправки)
 * <p>
 * Авторизация по токену аккаунта
 * </p>
 *
 * @see <a href="https://github.yandex-team.ru/sendr/sendr/blob/master/docs/transaction-api.md">Yandex Sender API</a>
 * @see <a href="https://wiki.yandex-team.ru/pochta/testirovanie/rassyljator/">Yandex Sender Homepage</a>
 */
public class YandexSenderClient {
    private static final Duration REQUEST_TIMEOUT = Duration.ofSeconds(8);
    private static final ObjectMapper MAPPER = new ObjectMapper();


    private static final String TO_EMAIL_PARAM = "to_email";
    private static final String INVALID_VALUE = "Invalid value";

    private final ParallelFetcherFactory parallelFetcherFactory;
    private final YandexSenderConfig senderConfig;

    public YandexSenderClient(YandexSenderConfig senderConfig, AsyncHttpClient asyncHttpClient) {
        this.senderConfig = senderConfig;

        FetcherSettings settings = new FetcherSettings()
                .withRequestTimeout(REQUEST_TIMEOUT);
        parallelFetcherFactory = new ParallelFetcherFactory(asyncHttpClient, settings);
    }

    private RequestBuilder createRequestBuilder(YandexSenderTemplateParams templateParams) {
        String url = "transactional/"+ templateParams.getCampaignSlug() +"/send";
        String requestUrl = getFullUrl(url);
        Param toParam = new Param(TO_EMAIL_PARAM, templateParams.getToEmail());

        Realm realm = new Realm.Builder(senderConfig.getAccountToken(), "")
                .setScheme(Realm.AuthScheme.BASIC)
                .setUsePreemptiveAuth(true)
                .build();

        RequestBuilder result = new RequestBuilder(POST)
                .setUrl(requestUrl)
                .addHeader(CONTENT_TYPE, APPLICATION_X_WWW_FORM_URLENCODED)
                .setRealm(realm)
                .addQueryParams(singletonList(toParam));

        if (templateParams.getAsync() != null) {
            result = result.addFormParam("async", templateParams.getAsync().toString());
        }

        if (templateParams.getArgs() != null) {
            try {
                result = result.addFormParam("args", MAPPER.writeValueAsString(templateParams.getArgs()));
            } catch (JsonProcessingException ex) {
                throw new YandexSenderException("can't serialize template args", ex);
            }
        }
        return result;
    }

    /**
     * Отправляем шаблон и при наличии ошибок в ответе кидаем исключение
     *
     * @param templateParams параметры шаблона
     */
    public void sendTemplate(YandexSenderTemplateParams templateParams) {
        sendTemplate(templateParams, r -> false);
    }

    /**
     * Отправляем письмо по шаблону.
     * При успешной отправке возвращаем true,
     * при неуспешной проверяем ответ рассылятора заданной функцией
     * и кидаем исключение, если она вернула ложное значение
     *
     * @param templateParams   параметры шаблона
     * @param canSuppressError функция для проверки наличия ошибки
     * @return успешна ли отправка
     */
    public boolean sendTemplate(YandexSenderTemplateParams templateParams,
                                Predicate<YandexSenderResultSection> canSuppressError) {
        checkNotNull(canSuppressError, "function must not be null");

        RequestBuilder builder = createRequestBuilder(templateParams);
        try (
                TraceProfile ignore = Trace.current().profile("email-sender", "send");
                ParallelFetcher<YandexSenderResponseEntity> fetcher = parallelFetcherFactory.getParallelFetcher()) {
            Result<YandexSenderResponseEntity> result;
            try {
                result = fetcher.execute(new YandexSenderResponseParser(0, builder.build()));
            } catch (InterruptedException ex) {
                Thread.currentThread().interrupt();
                throw new YandexSenderException("Mail sender exception", ex);
            }

            YandexSenderResponseEntity responseSuccessEntity = result.getSuccess();

            if (responseSuccessEntity == null) {
                throw new YandexSenderException("Mail sender exception", result.getErrors());
            }

            if (responseSuccessEntity.getResult().getStatus() == StatusEnum.OK) {
                return true;
            }

            if (canSuppressError.test(responseSuccessEntity.getResult())) {
                return false;
            } else {
                throw new YandexSenderException("Sender returns error: " + responseSuccessEntity.getResult().getError()
                        + "; message: " + responseSuccessEntity.getResult().getMessage());
            }
        }
    }

    private String getFullUrl(String url) {
        return senderConfig.getProtocol()
                + "://"
                + senderConfig.getSenderHost()
                + "/api/0/"
                + senderConfig.getAccountSlug()
                + "/"
                + url;
    }

    /**
     * Проверяет, есть ли ошибка валидации email в ответе от Рассылятора
     *
     * @param result результат запроса к Рассылятору
     * @return есть ли ошибка валидации email
     */
    public static boolean isInvalidToEmail(YandexSenderResultSection result) {
        Object error = result.getError();
        if (!(error instanceof Map)) {
            return false;
        }

        Object toEmailError = ((Map) error).get(TO_EMAIL_PARAM);

        if (!(toEmailError instanceof Collection)) {
            return false;
        }

        return ((Collection) toEmailError).contains(INVALID_VALUE);
    }

}
