package ru.yandex.direct.yav.client

import org.asynchttpclient.AsyncHttpClient
import ru.yandex.direct.asynchttp.FetcherSettings
import ru.yandex.direct.asynchttp.ParallelFetcherFactory
import ru.yandex.direct.http.smart.core.Smart
import ru.yandex.direct.liveresource.LiveResource
import java.util.concurrent.CompletableFuture
import java.util.concurrent.ExecutionException

/**
 * YavClientImpl - это клиент для Секретницы (https://yav.yandex-team.ru/)
 */
class YavClientImpl(private val asyncHttpClient: AsyncHttpClient,
                    fetcherSettings: FetcherSettings,
                    yavClientConfig: YavClientConfig,
                    tokenLiveResource: LiveResource) : YavClient {

    data class YavClientConfig(val baseUrl: String)

    private val yavClientApi = Smart.builder()
        .withParallelFetcherFactory(
            ParallelFetcherFactory(asyncHttpClient, fetcherSettings)
        )
        .withProfileName("yav_api")
        .withBaseUrl(yavClientConfig.baseUrl)
        .addHeaderConfigurator { headers -> headers.add("Content-type", "application/json") }
        .addHeaderConfigurator { headers ->
            val token = tokenLiveResource.content.trim()
            headers.add("Authorization", "OAuth $token")
        }
        .build()
        .create(Api::class.java)

    override fun getSecretValues(secretUuid: String): Map<String, String> {
        val versionResp = getSecretValuesFuture(secretUuid)

        try {
            return versionResp.get()
        } catch (e: ExecutionException) {
            throw e.cause ?: e
        }
    }

    override fun getSecretValuesFuture(secretUuid: String): CompletableFuture<Map<String, String>> {
        val versionRequest = yavClientApi.getVersion(secretUuid).request
        val versionAhcRequest = versionRequest.ahcRequest

        return asyncHttpClient.executeRequest(versionAhcRequest) // TODO: make retries
            .toCompletableFuture()
            .thenApply(versionRequest.parseFunction)
            .thenApply(::getKeyValueFromVersionResponse)
    }

    private fun getKeyValueFromVersionResponse(versionResp: SecretVersionResp): Map<String, String> {
        val values = versionResp.version.value
        return values.associate { it.key to it.decodedValue }
    }
}
