package ru.yandex.direct.ydb.builder.expression;

import java.util.ArrayList;
import java.util.List;

import com.yandex.ydb.table.values.PrimitiveType;
import com.yandex.ydb.table.values.Type;

import ru.yandex.direct.ydb.builder.YqlWithParams;
import ru.yandex.direct.ydb.builder.predicate.Predicate;
import ru.yandex.direct.ydb.builder.valuecreator.PrimitiveValueCreator;
import ru.yandex.direct.ydb.builder.valuecreator.TypeValueMapper;
import ru.yandex.direct.ydb.builder.valuecreator.ValueCreator;

import static ru.yandex.direct.ydb.builder.valuecreator.TypeValueMapper.getPrimitiveCreator;

public class CaseExpression<T> implements Expression<T> {

    private final Type type;
    private final ValueCreator<T> valueFunction;
    private final List<YqlWithParams> yqlWithParamsList = new ArrayList<>();

    private CaseExpression(List<YqlWithParams> whenYqlWithParamsList, Type type, ValueCreator<T> valueFunction) {
        this.type = type;
        this.valueFunction = valueFunction;
        yqlWithParamsList.add(new YqlWithParams("CASE "));
        yqlWithParamsList.addAll(whenYqlWithParamsList);
        yqlWithParamsList.add(new YqlWithParams(" END"));
    }

    @Override
    public Type getType() {
        return type;
    }

    @Override
    public ValueCreator<T> getValueCreator() {
        return valueFunction;
    }

    @Override
    public List<YqlWithParams> getYqlWithParamsList() {
        return yqlWithParamsList;
    }

    public static <T> WhenStatement<T> caseWhen(Predicate predicate, Expression<T> expression) {
        return new WhenStatement<>(predicate, expression);
    }

    public static <T> WhenStatement<T> caseWhen(Predicate predicate, T value, PrimitiveType type) {
        PrimitiveValueCreator<T> primitiveValueCreator = getPrimitiveCreator(type);
        return new WhenStatement<>(predicate, new ConstantExpression<>(value, type, primitiveValueCreator::toYdb));
    }

    public static <T> WhenStatement<T> caseWhen(Predicate predicate, T value, Type type) {
        ValueCreator<T> primitiveValueCreator = TypeValueMapper.getValueCreator(type);
        return new WhenStatement<>(predicate, new ConstantExpression<>(value, type, primitiveValueCreator::toYdb));
    }

    public static <T> WhenStatement<T> caseWhen(Predicate predicate, T value, Type type,
                                                ValueCreator<T> valueCreator) {
        return new WhenStatement<>(predicate, new ConstantExpression<>(value, type, valueCreator));
    }

    public static class WhenStatement<T> {
        private final Type type;
        private final ValueCreator<T> valueFunction;
        private final List<YqlWithParams> yqlWithParamsList = new ArrayList<>();

        WhenStatement(Predicate predicate, Expression<T> expression) {
            this.type = expression.getType();
            this.valueFunction = expression.getValueCreator();
            addWhenStatement(predicate, expression);
        }

        private void addWhenStatement(Predicate predicate, Expression<T> expression) {
            yqlWithParamsList.add(new YqlWithParams("WHEN "));
            yqlWithParamsList.addAll(predicate.getYqlWithParamsList());
            yqlWithParamsList.add(new YqlWithParams(" THEN "));
            yqlWithParamsList.addAll(expression.getYqlWithParamsList());
            yqlWithParamsList.add(new YqlWithParams("\n"));
        }

        private void addElseStatement(Expression<T> expression) {
            yqlWithParamsList.add(new YqlWithParams("ELSE "));
            yqlWithParamsList.addAll(expression.getYqlWithParamsList());
            yqlWithParamsList.add(new YqlWithParams("\n"));
        }

        public WhenStatement<T> when(Predicate predicate, Expression<T> expression) {
            addWhenStatement(predicate, expression);
            return this;
        }

        public WhenStatement<T> when(Predicate predicate, T value) {
            var expression = new ConstantExpression<>(value, type, valueFunction);
            return when(predicate, expression);
        }

        public CaseExpression<T> otherwise(Expression<T> expression) {
            addElseStatement(expression);
            return new CaseExpression<>(yqlWithParamsList, type, valueFunction);
        }

        public CaseExpression<T> otherwise(T value) {
            var expression = new ConstantExpression<>(value, type, valueFunction);
            return otherwise(expression);
        }
    }
}
