package ru.yandex.direct.ydb.builder.expression;

import java.util.Arrays;
import java.util.Collection;
import java.util.Objects;

import javax.annotation.Nonnull;

import com.yandex.ydb.table.values.PrimitiveType;
import com.yandex.ydb.table.values.Type;

import ru.yandex.direct.ydb.builder.YqlQueryHelper;
import ru.yandex.direct.ydb.builder.predicate.ComparablePredicate;
import ru.yandex.direct.ydb.builder.predicate.IsNotNullPredicate;
import ru.yandex.direct.ydb.builder.predicate.IsNullPredicate;
import ru.yandex.direct.ydb.builder.predicate.LikePredicate;
import ru.yandex.direct.ydb.builder.predicate.Predicate;
import ru.yandex.direct.ydb.builder.valuecreator.TypeValueMapper;
import ru.yandex.direct.ydb.builder.valuecreator.ValueCreator;
import ru.yandex.direct.ydb.column.Column;

public interface Expression<T> extends YqlQueryHelper {
    Type getType();

    ValueCreator<T> getValueCreator();

    default Predicate eq(T value) {
        if (Objects.isNull(value)) {
            return isNull();
        }

        var param = this instanceof NamedExpression ? ((NamedExpression) this).getName() : "param";
        var constExp = new ConstantExpression<>(value, param, getType(), getValueCreator());
        return ComparablePredicate.eq(this, constExp);
    }

    default Predicate neq(T value) {
        if (Objects.isNull(value)) {
            return isNotNull();
        }
        var param = this instanceof NamedExpression ? ((NamedExpression) this).getName() : "param";
        var constExp = new ConstantExpression<>(value, param, getType(), getValueCreator());
        return ComparablePredicate.ne(this, constExp);
    }

    default Predicate lt(@Nonnull T value) {
        var param = this instanceof NamedExpression ? ((NamedExpression) this).getName() : "param";
        var constExp = new ConstantExpression<>(value, param, getType(), getValueCreator());
        return ComparablePredicate.lt(this, constExp);
    }

    default Predicate gt(@Nonnull T value) {
        var param = this instanceof NamedExpression ? ((NamedExpression) this).getName() : "param";
        var constExp = new ConstantExpression<>(value, param, getType(), getValueCreator());
        return ComparablePredicate.gt(this, constExp);
    }

    default Predicate le(@Nonnull T value) {
        var param = this instanceof NamedExpression ? ((NamedExpression) this).getName() : "param";
        var constExp = new ConstantExpression<>(value, param, getType(), getValueCreator());
        return ComparablePredicate.le(this, constExp);
    }

    default Predicate ge(@Nonnull T value) {
        var param = this instanceof NamedExpression ? ((NamedExpression) this).getName() : "param";
        var constExp = new ConstantExpression<>(value, param, getType(), getValueCreator());
        return ComparablePredicate.ge(this, constExp);
    }

    default Predicate in(@Nonnull Collection<T> values) {
        if (values.size() == 1) {
            return eq(values.iterator().next());
        }
        var param = this instanceof NamedExpression ? ((NamedExpression) this).getName() : "param";
        var constExp = new ListExpression<>(values, param, getType(), getValueCreator());
        return ComparablePredicate.in(this, constExp);
    }

    default Predicate in(T... values) {
        return in(Arrays.asList(values));
    }

    default Predicate eq(@Nonnull Expression<T> another) {
        return ComparablePredicate.eq(this, another);
    }

    default Predicate neq(@Nonnull Column<T> another) {
        return ComparablePredicate.ne(this, another);
    }

    default Predicate lt(@Nonnull Column<T> another) {
        return ComparablePredicate.lt(this, another);
    }

    default Predicate gt(@Nonnull Column<T> another) {
        return ComparablePredicate.gt(this, another);
    }

    default Predicate le(@Nonnull Column<T> another) {
        return ComparablePredicate.le(this, another);
    }

    default Predicate ge(@Nonnull Column<T> another) {
        return ComparablePredicate.ge(this, another);
    }

    default Predicate isNull() {
        return new IsNullPredicate(this);
    }

    default Predicate isNotNull() {
        return new IsNotNullPredicate(this);
    }

    default <R> Expression<R> minus(Expression<T> another, PrimitiveType resultType) {
        return ArithmeticExpression.minus(this, another, resultType);
    }

    default <R> Expression<R> plus(Expression<T> another, PrimitiveType resultType) {
        return ArithmeticExpression.plus(this, another, resultType);
    }

    /**
     * case sensitive
     **/
    default Predicate like(Expression<String> another) {
        return LikePredicate.like(this, another);
    }

    /**
     * case insensitive
     **/
    default Predicate ilike(Expression<String> another) {
        return LikePredicate.ilike(this, another);
    }

    /**
     * case sensitive
     **/
    default Predicate like(String value) {
        var param = this instanceof NamedExpression ? ((NamedExpression) this).getName() : "param";
        var constExp = new ConstantExpression<>(value, param, getType(),
                TypeValueMapper.getPrimitiveCreator(getType()));
        return LikePredicate.like(this, constExp);
    }

    /**
     * case insensitive
     **/
    default Predicate ilike(String value) {
        var param = this instanceof NamedExpression ? ((NamedExpression) this).getName() : "param";
        var constExp = new ConstantExpression<>(value, param, getType(),
                TypeValueMapper.getPrimitiveCreator(getType()));
        return LikePredicate.ilike(this, constExp);
    }

    default AliasedExpression<T> as(@Nonnull String name) {
        return new AliasedExpression<>(this, name);
    }
}
