package ru.yandex.direct.ydb.builder.querybuilder;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import ru.yandex.direct.ydb.builder.YqlWithParams;
import ru.yandex.direct.ydb.builder.expression.Expression;
import ru.yandex.direct.ydb.builder.predicate.ComparablePredicate;
import ru.yandex.direct.ydb.builder.predicate.Predicate;
import ru.yandex.direct.ydb.table.TableExpression;

public class JoinBuilder extends QueryBuilder implements SubQueryLike, UnionLike {
    private final List<YqlWithParams> yqlWithParamsList = new ArrayList<>();

    JoinBuilder(QueryBuilder subBuilder, TableExpression tableExpression, JoinStatement joinStatement) {
        this(false, subBuilder, tableExpression, joinStatement);
    }

    JoinBuilder(boolean isLeftJoin,
                QueryBuilder subBuilder,
                TableExpression tableExpression,
                JoinStatement joinStatement) {
        yqlWithParamsList.addAll(subBuilder.getYqlWithParamsList());
        yqlWithParamsList.add(new YqlWithParams((isLeftJoin ? "LEFT " : "") + "JOIN "));
        yqlWithParamsList.add(new YqlWithParams(tableExpression.getTableDeclare(), tableExpression.getParams()));
        yqlWithParamsList.add(new YqlWithParams(" ON "));
        yqlWithParamsList.addAll(joinStatement.joinPredicate.getYqlWithParamsList());
        yqlWithParamsList.add(new YqlWithParams("\n"));
    }

    JoinBuilder(QueryBuilder subBuilder, SubQueryBuilder subQueryBuilder, JoinStatement joinStatement) {
        yqlWithParamsList.addAll(subQueryBuilder.getYqlWithParamsList());
        yqlWithParamsList.add(new YqlWithParams("JOIN "));
        yqlWithParamsList.addAll(subQueryBuilder.getYqlWithParamsList());
        yqlWithParamsList.add(new YqlWithParams(" ON "));
        yqlWithParamsList.addAll(joinStatement.joinPredicate.getYqlWithParamsList());
        yqlWithParamsList.add(new YqlWithParams("\n"));
    }

    public static class JoinStatement {

        private Predicate joinPredicate;

        JoinStatement(Predicate startPredicate) {
            this.joinPredicate = startPredicate;
        }

        public static <T> JoinStatement on(Expression<T> expression1, Expression<T> expression2) {
            return new JoinStatement(ComparablePredicate.eq(expression1, expression2));
        }

        public <T> JoinStatement and(Expression<T> expression1, Expression<T> expression2) {
            return new JoinStatement(joinPredicate.and(ComparablePredicate.eq(expression1, expression2)));
        }
    }

    @Override
    public List<YqlWithParams> getYqlWithParamsList() {
        return yqlWithParamsList;
    }

    public ExtendedWhereBuilder where(Predicate predicate) {
        return new ExtendedWhereBuilder(this, predicate);
    }

    public GroupByBuilder groupBy(Expression... expressions) {
        List<Expression> expressionList = Arrays.asList(expressions);
        return new GroupByBuilder(this, expressionList);
    }

    public LimitBuilder limit(long limit) {
        return new LimitBuilder(this, limit);
    }

    public OffsetBuilder offset(long offset) {
        return new OffsetBuilder(this, offset);
    }

    public HavingBuilder having(Predicate predicate) {
        return new HavingBuilder(this, predicate);
    }

    public OrderByBuilder orderBy(OrderByBuilder.OrderType orderType, Expression expression1,
                                  Expression... expressions) {
        return new OrderByBuilder(this, orderType, expression1, expressions);
    }

    public OrderByBuilder orderBy(Expression expression1, Expression... expressions) {
        return orderBy(OrderByBuilder.OrderType.ASC, expression1, expressions);
    }

    public JoinBuilder join(TableExpression tableExpression, JoinBuilder.JoinStatement joinStatement) {
        return new JoinBuilder(this, tableExpression, joinStatement);
    }
}
