package ru.yandex.direct.ydb.builder.querybuilder;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicLong;

import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.values.Value;

import ru.yandex.direct.ydb.YdbPath;
import ru.yandex.direct.ydb.YqlVersion;
import ru.yandex.direct.ydb.builder.QueryAndParams;
import ru.yandex.direct.ydb.builder.TypeToStringConverter;
import ru.yandex.direct.ydb.builder.TypeToStringConverterV0;
import ru.yandex.direct.ydb.builder.TypeToStringConverterV1;
import ru.yandex.direct.ydb.builder.YqlQueryHelper;
import ru.yandex.direct.ydb.builder.YqlWithParams;

public abstract class QueryBuilder implements YqlQueryHelper {
    private static final String DECLARED_PARAM_TEMPLATE = "%s_%d";
    private static final String DECLARE_TEMPLATE = "DECLARE %s AS %s;\n";
    private static final String PRAGMA_TEMPLATE = "PRAGMA TablePathPrefix(\"%s\");%n";

    public QueryAndParams queryAndParams(YdbPath path) {
        return queryAndParams(path, YqlVersion.SQLv1);
    }

    public QueryAndParams queryAndParams(YdbPath path, YqlVersion yqlVersion) {
        TypeToStringConverter typeToStringConverter;
        if (yqlVersion == YqlVersion.SQLv0) {
            typeToStringConverter = new TypeToStringConverterV0();
        } else {
            typeToStringConverter = new TypeToStringConverterV1();
        }

        StringBuilder declareBuilder = new StringBuilder();
        Map<String, AtomicLong> paramNameToIndex = new HashMap<>();
        Map<String, Value<?>> ydbParamsToValue = new HashMap<>();
        var yqlWithParamsList = getYqlWithParamsList();
        StringBuilder yqlBuilder = new StringBuilder();
        for (var yqlWithParams : yqlWithParamsList) {
            String currentYql = yqlWithParams.getYql();
            var params = yqlWithParams.getParamList();
            for (var param : params) {
                var nextIndex =
                        paramNameToIndex.computeIfAbsent(param.getYdbParamName(), p -> new AtomicLong()).incrementAndGet();
                var paramNameWithIndex = String.format(DECLARED_PARAM_TEMPLATE, param.getYdbParamName(), nextIndex);
                declareBuilder.append(String.format(DECLARE_TEMPLATE, paramNameWithIndex,
                        typeToStringConverter.convert(param.getType())));
                currentYql = currentYql.replace(param.getYdbParamName(), paramNameWithIndex);
                ydbParamsToValue.put(paramNameWithIndex, param.getValue());
            }
            yqlBuilder.append(currentYql);
        }

        var query = String.format(PRAGMA_TEMPLATE, path.getPath()) + declareBuilder.toString() + yqlBuilder.toString();
        if (yqlVersion == YqlVersion.SQLv1) {
            query = "--!syntax_v1\n" + query;
        }
        var ydbParams = Params.copyOf(ydbParamsToValue);
        return new QueryAndParams(path.getPath(), query, ydbParams, detectQueryType(yqlWithParamsList));
    }

    // определяем тип запроса для трейсинга
    private QueryAndParams.Type detectQueryType(List<YqlWithParams> yqlWithParamsList) {
        var type = QueryAndParams.Type.UNKNOWN;

        for (var yqlWithParams : yqlWithParamsList) {
            String currentYql = yqlWithParams.getYql();
            if (type == QueryAndParams.Type.UNKNOWN && currentYql.startsWith("SELECT")) {
                type = QueryAndParams.Type.READ;
            } else if (
                    currentYql.startsWith("UPDATE")
                    || currentYql.startsWith("DELETE")
                    || currentYql.startsWith("INSERT")
            ) {
                type = QueryAndParams.Type.WRITE;
            }
        }

        return type;
    }


}
