package ru.yandex.direct.ydb.builder.querybuilder;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

import javax.annotation.Nonnull;

import com.yandex.ydb.table.values.Type;

import ru.yandex.direct.ydb.builder.YqlWithParams;
import ru.yandex.direct.ydb.builder.expression.ConstantExpression;
import ru.yandex.direct.ydb.builder.expression.Expression;
import ru.yandex.direct.ydb.builder.predicate.Predicate;
import ru.yandex.direct.ydb.builder.valuecreator.ValueCreator;
import ru.yandex.direct.ydb.column.Column;
import ru.yandex.direct.ydb.table.Table;

public class UpdateBuilder extends QueryBuilder {
    private final List<YqlWithParams> yqlWithParamsList = new ArrayList<>();

    private UpdateBuilder(Table table, @Nonnull SetStatement setStatement) {
        this.yqlWithParamsList.add(new YqlWithParams("UPDATE "));
        this.yqlWithParamsList.add(new YqlWithParams(table.getTableDeclare(), table.getParams()));
        this.yqlWithParamsList.add(new YqlWithParams("\n"));
        this.yqlWithParamsList.add(new YqlWithParams("SET "));
        this.yqlWithParamsList.addAll(setStatement.getYqlWithParamsList());
        this.yqlWithParamsList.add(new YqlWithParams("\n"));

    }

    @Override
    public List<YqlWithParams> getYqlWithParamsList() {
        return yqlWithParamsList;
    }

    public static UpdateBuilder update(Table table, SetStatement setStatement) {
        return new UpdateBuilder(table, setStatement);
    }

    public static <T> SetStatement set(Column<T> column, T value) {
        return new SetStatement(column, value);
    }

    public static <T> SetStatement set(Column<T> column, Expression<T> expression) {
        return new SetStatement(column, expression);
    }

    public LimitedWhereBuilder where(Predicate predicate) {
        return new LimitedWhereBuilder(this, predicate);
    }

    public static class SetStatement {
        private static final YqlWithParams DELIMITER = new YqlWithParams(",\n");
        private final List<YqlWithParams> yqlWithParamsList = new ArrayList<>();

        private <T> SetStatement(Column<T> column, Expression<T> expression) {
            addStatement(column, expression);
        }

        private <T> SetStatement(Column<T> column, T value) {
            set(column, value, false);
        }

        public <T> SetStatement set(Column<T> column, Expression<T> expression) {
            yqlWithParamsList.add(DELIMITER);
            addStatement(column, expression);
            return this;
        }

        public <T> SetStatement set(Column<T> column, T value) {
            return set(column, value, true);
        }

        private <T> SetStatement set(Column<T> column, T value, boolean addDelimiter) {
            if (addDelimiter) {
                yqlWithParamsList.add(DELIMITER);
            }
            Type resType;
            ValueCreator<T> ydbValueCreator;
            if (Objects.isNull(value)) {
                var optionalType = column.getType().makeOptional();
                resType = optionalType;
                ydbValueCreator = v -> optionalType.emptyValue();
            } else {
                resType = column.getType();
                ydbValueCreator = column.getValueCreator();
            }
            addStatement(column, new ConstantExpression<>(value, column.getColumnName(), resType, ydbValueCreator));
            return this;
        }

        public <T> SetStatement setNull(Column<T> column) {
            yqlWithParamsList.add(DELIMITER);
            ValueCreator<T> ydbValueCreator;
            var optionalType = column.getType().makeOptional();
            ydbValueCreator = v -> optionalType.emptyValue();
            addStatement(column, new ConstantExpression<>(null, column.getColumnName(), optionalType, ydbValueCreator));
            return this;
        }

        private <T> void addStatement(Column<T> column, Expression<T> expression) {
            yqlWithParamsList.add(new YqlWithParams(column.getName()));
            yqlWithParamsList.add(new YqlWithParams(" = "));
            yqlWithParamsList.addAll(expression.getYqlWithParamsList());
        }

        private List<YqlWithParams> getYqlWithParamsList() {
            return yqlWithParamsList;
        }
    }
}
