package ru.yandex.direct.ydb.client;

import java.time.Duration;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.function.Consumer;

import com.yandex.ydb.core.Status;
import com.yandex.ydb.table.SessionRetryContext;
import com.yandex.ydb.table.settings.ExecuteDataQuerySettings;
import com.yandex.ydb.table.settings.ReadTableSettings;
import com.yandex.ydb.table.transaction.TxControl;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.utils.InterruptedRuntimeException;
import ru.yandex.direct.ydb.YdbPath;
import ru.yandex.direct.ydb.builder.QueryAndParams;
import ru.yandex.direct.ydb.exceptions.YdbExecutionQueryException;
import ru.yandex.direct.ydb.table.Table;

import static com.yandex.ydb.table.transaction.TxControl.serializableRw;

public class YdbClient {
    private final static TxControl<?> DEFAULT_TX_CONTROL = serializableRw();

    private final SessionRetryContext sessionRetryContext;
    private final Duration queryTimeout;

    private static final Logger logger = LoggerFactory.getLogger(YdbClient.class);

    public YdbClient(SessionRetryContext sessionRetryContext, Duration queryTimeout) {
        this.sessionRetryContext = sessionRetryContext;
        this.queryTimeout = queryTimeout;
    }

    public CompletableFuture<DataQueryResultWrapper> executeQueryAsync(QueryAndParams queryAndParams) {
        return executeQueryAsync(queryAndParams, false);
    }

    public CompletableFuture<DataQueryResultWrapper> executeQueryAsync(QueryAndParams queryAndParams,
                                                                       boolean keepQueryInCache) {
        return executeQueryAsync(queryAndParams,
                "Failed to execute query " + queryAndParams.getQuery() + " " + queryAndParams.getParams().toPb(),
                keepQueryInCache);
    }

    public DataQueryResultWrapper executeQuery(QueryAndParams queryAndParams) {
        return executeQuery(queryAndParams, false);

    }

    public DataQueryResultWrapper executeQuery(QueryAndParams queryAndParams, boolean keepQueryInCache) {
        return executeQuery(queryAndParams,
                "Failed to execute query " + queryAndParams.getQuery() + " " + queryAndParams.getParams(),
                keepQueryInCache);

    }

    public DataQueryResultWrapper executeOnlineRoQuery(QueryAndParams queryAndParams, boolean keepQueryInCache) {
        return executeQuery(queryAndParams,
                "Failed to execute query " + queryAndParams.getQuery() + " " + queryAndParams.getParams(),
                keepQueryInCache, TxControl.onlineRo());

    }

    public DataQueryResultWrapper executeOnlineRoQuery(QueryAndParams queryAndParams, boolean keepQueryInCache,
                                                       Duration timeout) {
        return executeQuery(queryAndParams,
                "Failed to execute query " + queryAndParams.getQuery() + " " + queryAndParams.getParams(),
                keepQueryInCache, TxControl.onlineRo(), timeout);

    }

    public DataQueryResultWrapper executeQuery(QueryAndParams queryAndParams, String errorMessage) {
        return executeQuery(queryAndParams, errorMessage, false);
    }

    public DataQueryResultWrapper executeQuery(QueryAndParams queryAndParams, String errorMessage,
                                               boolean keepQueryInCache) {
        return executeQuery(queryAndParams, errorMessage, keepQueryInCache, DEFAULT_TX_CONTROL);
    }

    public DataQueryResultWrapper executeQuery(QueryAndParams queryAndParams, String errorMessage,
                                               boolean keepQueryInCache, TxControl<?> txControl) {
        return executeQuery(queryAndParams, errorMessage, keepQueryInCache, txControl, queryTimeout);
    }

    public DataQueryResultWrapper executeQuery(QueryAndParams queryAndParams, String errorMessage,
                                               boolean keepQueryInCache, TxControl<?> txControl, Duration timeout) {
        try (var unused = Trace.current().profile(
                "ydb:" + queryAndParams.getType().toString().toLowerCase(),
                formatPath(queryAndParams.getPath()))
        ) {
            return executeQueryAsync(queryAndParams, errorMessage, keepQueryInCache, txControl, timeout)
                    .get(timeout.getSeconds(), TimeUnit.SECONDS);
        } catch (InterruptedException ex) {
            Thread.currentThread().interrupt();
            throw new InterruptedRuntimeException(ex);
        } catch (ExecutionException | TimeoutException ex) {
            throw new YdbExecutionQueryException(ex);
        }
    }

    // форматирование пути к БД ydb для трейсинга
    private String formatPath(String path) {
        path = StringUtils.stripStart(path, "/");
        path = path.replace('/', '.');
        return path;
    }

    public CompletableFuture<DataQueryResultWrapper> executeQueryAsync(QueryAndParams queryAndParams,
                                                                       String errorMessage) {
        return executeQueryAsync(queryAndParams, errorMessage, false);
    }

    public CompletableFuture<DataQueryResultWrapper> executeQueryAsync(QueryAndParams queryAndParams,
                                                                       String errorMessage, boolean keepQueryInCache) {
        return executeQueryAsync(queryAndParams, errorMessage, keepQueryInCache, DEFAULT_TX_CONTROL);
    }

    public CompletableFuture<DataQueryResultWrapper> executeQueryAsync(QueryAndParams queryAndParams,
                                                                       String errorMessage, boolean keepQueryInCache,
                                                                       TxControl<?> txControl) {

        return executeQueryAsync(queryAndParams, errorMessage, keepQueryInCache, txControl, queryTimeout);
    }

    public CompletableFuture<DataQueryResultWrapper> executeQueryAsync(QueryAndParams queryAndParams,
                                                                       String errorMessage, boolean keepQueryInCache,
                                                                       TxControl<?> txControl, Duration timeout) {

        var executingDataQuerySettings = new ExecuteDataQuerySettings();
        if (keepQueryInCache) {
            executingDataQuerySettings.keepInQueryCache();
        }

        executingDataQuerySettings.setTimeout(timeout);
        logger.debug("Ydb query {}, params {}", queryAndParams.getQuery(), queryAndParams.getParams().toPb());
        return sessionRetryContext.supplyResult(session -> session.executeDataQuery(queryAndParams.getQuery(),
                txControl, queryAndParams.getParams(), executingDataQuerySettings))
                .thenApply(dataQueryResultResult -> dataQueryResultResult.expect(errorMessage))
                .thenApply(DataQueryResultWrapper::new);
    }

    public CompletableFuture<Status> readTable(YdbPath path, Table table, Duration timeout,
                                               Consumer<ResultSetReaderWrapped> resultSetReaderWrappedConsumer) {
        logger.debug("Ydb read table {}", table.getRealName());
        return sessionRetryContext.supplyStatus(session -> session.readTable(path.getPath() + table.getRealName(),
                ReadTableSettings.newBuilder().timeout(timeout).build(), resultSetReader -> {
                    var resultSetReaderWrapped = new ResultSetReaderWrapped(resultSetReader);
                    while (resultSetReader.next()) {
                        resultSetReaderWrappedConsumer.accept(resultSetReaderWrapped);
                    }
                }));
    }
}
