package ru.yandex.direct.ytwrapper;

import java.time.Instant;
import java.util.function.Function;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.jooq.Field;
import org.jooq.TableField;

import ru.yandex.yt.ytclient.tables.TableSchema;
import ru.yandex.yt.ytclient.wire.UnversionedRow;
import ru.yandex.yt.ytclient.wire.UnversionedRowset;
import ru.yandex.yt.ytclient.wire.UnversionedValue;

import static com.google.common.base.Preconditions.checkArgument;

/**
 * вспомогательные функции для работы с YT таблицами
 */
public class YtTableUtils {

    /**
     * Алиас для YT-запросов. YT не понимает запросов, где в select list'е и условии группировки
     * используются колонки без объявленных alias'ов.
     * Не использовать для полей под аггрегирующей функцией.
     */
    public static <T> Field<T> aliased(TableField<?, T> tableField) {
        return tableField.as(tableField.getName());
    }

    /**
     * Возвращает индекс колонки {@param field} в {@param tableSchema}.
     *
     * @throws IllegalArgumentException если колонка не найдена
     */
    public static int findColumnOrThrow(TableSchema tableSchema, String field) {
        int idx = tableSchema.findColumn(field);
        checkArgument(idx != -1, "Can't find column '%s'", field);
        return idx;
    }

    public static int findColumnOrThrow(TableSchema tableSchema, Field<?> field) {
        return findColumnOrThrow(tableSchema, field.getName());
    }

    /**
     * Возвращает EntryStream: значение поля {@param key} в строке из {@param rowset} -> результат применения
     * {@param valueFunc} к строке из {@param rowset}
     */
    public static <T> EntryStream<UnversionedValue, T> toEntryStream(UnversionedRowset rowset, Field keyField,
                                                                     Function<UnversionedRow, T> valueMapper) {
        TableSchema schema = rowset.getSchema();
        return StreamEx.of(rowset.getRows()).mapToEntry(fieldValueGetter(schema, keyField), valueMapper);
    }

    /**
     * Возвращает EntryStream: значение поля {@param keyField} -> значение поля {@param valueField}
     * из значений в {@param rowset}
     */
    public static EntryStream<UnversionedValue, UnversionedValue> toEntryStream(
            UnversionedRowset rowset, Field keyField, Field valueField) {
        return toEntryStream(rowset, keyField, fieldValueGetter(rowset.getSchema(), valueField));
    }

    /**
     * Возвращает функцию, которая возращает значение поля {@param field} для {@link UnversionedRow}
     */
    public static Function<UnversionedRow, UnversionedValue> fieldValueGetter(TableSchema schema, Field field) {
        return row -> row.getValues().get(findColumnOrThrow(schema, field));
    }

    public static Function<UnversionedRow, Long> longValueGetter(
            TableSchema schema, Field<? extends Number> longField) {
        return fieldValueGetter(schema, longField).andThen(nullSafetyMapper(UnversionedValue::longValue));
    }

    public static Function<UnversionedRow, Boolean> booleanValueGetter(
            TableSchema schema, Field<Boolean> booleanField) {
        return fieldValueGetter(schema, booleanField).andThen(nullSafetyMapper(UnversionedValue::booleanValue));
    }

    public static Function<UnversionedRow, String> stringValueGetter(
            TableSchema schema, Field<String> stringField) {
        return fieldValueGetter(schema, stringField).andThen(nullSafetyMapper(UnversionedValue::stringValue));
    }

    public static Function<UnversionedRow, byte[]> bytesValueGetter(
            TableSchema schema, Field<String> bytesField) {
        return fieldValueGetter(schema, bytesField).andThen(nullSafetyMapper(UnversionedValue::bytesValue));
    }

    public static Function<UnversionedRow, Instant> instantValueGetter(
            TableSchema schema, Field<Long> instantField) {
        return fieldValueGetter(schema, instantField).andThen(UnversionedValue::longValue)
                .andThen(Instant::ofEpochSecond);
    }

    public static <T> Function<UnversionedValue, T> nullSafetyMapper(Function<UnversionedValue, T> mapper) {
        return v -> v.getValue() != null ? mapper.apply(v) : null;
    }
}
