package ru.yandex.direct.ytwrapper;

import java.time.LocalDate;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Consumer;

import javax.annotation.Nullable;

import com.google.common.collect.Sets;
import org.apache.commons.lang3.tuple.Pair;
import org.jooq.Condition;
import org.jooq.Field;
import org.jooq.impl.DSL;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.dynamic.dsl.YtDSL;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.cypress.RangeLimit;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.common.YtServiceUnavailableException;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTree;
import ru.yandex.inside.yt.kosher.tables.YTableEntryTypes;
import ru.yandex.inside.yt.kosher.ytree.YTreeMapNode;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;

import static java.util.function.Function.identity;
import static java.util.stream.Collectors.toMap;

/**
 * Просто вспомогательные функции, которые не очень удобно было бы включать напрямую в клиенты/обёртки.
 */
public class YtUtils {

    public static final String FASTBONE = "fastbone";
    public static final String COMPRESSION_RATIO_ATTR = "compression_ratio";
    public static final String COMPRESSION_CODEC_ATTR = "compression_codec";
    public static final String ERASURE_CODEC_ATTR = "erasure_codec";
    public static final String DATA_SIZE_PER_JOB_ATTR = "data_size_per_job";
    public static final String EXPIRATION_TIME_ATTR = "expiration_time";
    public static final String SCHEMA_ATTR = "schema";
    public static final String DATA_WEIGHT_ATTR = "data_weight";
    public static final String COMPRESSED_DATA_SIZE = "compressed_data_size";
    public static final String STATE_ATTR = "state";
    public static final String SORTED_ATTR = "sorted";
    public static final String CONTENT_REVISION_ATTR = "content_revision";

    public static final String CHUNK_ROW_COUNT_ATTR = "chunk_row_count";
    public static final String DATA_SIZE_ATTR = "compressed_data_size";

    // атрибут в который записываем время последнего апдейта таблицы
    public static final String LAST_UPDATE_TIME_ATTR = "last_update_time";

    /**
     * Возвращает resource usage для указанной ноды в разбивке по медиумам.
     * Если для какого-то medium не возвращается значения, это означает, что resource usage для него = 0.
     */
    public static Map<String, Long> getRecursiveUsageByMediums(Yt yt, YPath path) {
        YTreeNode resourceUsage = yt.cypress().get(path, Cf.set("recursive_resource_usage"));
        YTreeNode diskSpacePerMedium = resourceUsage
                .getAttributeOrThrow("recursive_resource_usage")
                .asMap()
                .get("disk_space_per_medium");
        return Cf.wrap(diskSpacePerMedium.asMap()).mapValues(YTreeNode::longValue);
    }

    /**
     * Возвращает количество свободного пространства на кластере cluster для указанного аккаунта account
     * (в разбивке по медиумам). Если какой-то ресурс присутствует только в одном из атрибутов
     * resource_usage и resource_limits, то он не будет выведен.
     */
    public static Map<String, Long> getRemainingSpaceByMediums(Yt yt, String account) {
        Map<String, YTreeNode> accountInfo = yt.cypress().get(
                YPath.simple(String.format("//sys/accounts/%s/@", account)),
                Cf.set("resource_usage", "resource_limits"))
                .asMap();
        Map<String, YTreeNode> resourceUsage =
                accountInfo.get("resource_usage").asMap().get("disk_space_per_medium").asMap();
        Map<String, YTreeNode> resourceLimits =
                accountInfo.get("resource_limits").asMap().get("disk_space_per_medium").asMap();
        return Sets.intersection(resourceUsage.keySet(), resourceLimits.keySet()).stream()
                .collect(toMap(identity(), medium -> {
                    long limit = resourceLimits.get(medium).longValue();
                    long usage = resourceUsage.get(medium).longValue();
                    return Math.max(0L, limit - usage);
                }));
    }

    /**
     * Возвращает имя аккаунта в YT, который используется для директории dir
     */
    public static String getYtAccount(Yt yt, YPath dir) {
        return yt.cypress().get(dir, Cf.set("account")).getAttributeOrThrow("account").stringValue();
    }

    public static boolean isClusterAvailable(YtProvider ytProvider, YtCluster cluster) {
        boolean success = true;
        try {
            ytProvider.get(cluster).cypress().exists(YPath.cypressRoot());
        } catch (YtServiceUnavailableException ignored) {
            success = false;
        }
        return success;
    }

    public static TraceProfile createRpcProfile(YtCluster cluster) {
        return Trace.current().profile("yt.rpc", cluster.getName());
    }

    public static Field<Long> effectiveCampaignIdField(Field<Long> campaignIdField,
                                                       @Nullable Map<Long, Long> masterIdBySubId) {
        if (masterIdBySubId == null || masterIdBySubId.isEmpty()) {
            return campaignIdField;
        }
        var masterCampaignIdField = DSL.field("masterCampaignId", Long.class);
        return YtDSL.ytIf(
                YtDSL.isNull(
                        YtDSL.alias(
                                YtDSL.transform(campaignIdField, masterIdBySubId), masterCampaignIdField
                        )
                ), campaignIdField, masterCampaignIdField
        );
    }

    public static Condition periodCondition(Field<Long> updateTime, LocalDate startDate, LocalDate endDate) {
        List<Field<Long>> periods = new ArrayList<>();
        for (LocalDate date = startDate; date.isBefore(endDate) || date.equals(endDate); date = date.plusDays(1)) {
            periods.add(YtDSL.toEpochSecondsAtStartOfDate(date));
        }

        if (periods.size() == 1) {
            return updateTime.eq(periods.get(0));
        }

        return updateTime.in(periods);
    }

    public static Condition betweenRange(Field<Long> updateTime, LocalDate startDate, LocalDate endDate) {
        var startUpdateTime = YtDSL.toEpochSecondsAtStartOfDate(startDate);
        var endUpdateTime = YtDSL.toEpochSecondsAtStartOfDate(endDate);
        return updateTime.between(startUpdateTime, endUpdateTime);
    }

    public static void readTableByKeyRanges(Yt yt,
                                            YPath path,
                                            List<Pair<Long, Long>> ranges,
                                            Consumer<YTreeMapNode> consumer) {
        for (var range : ranges) {
            path = path.withRange(
                    new RangeLimit(Cf.list(YTree.longNode(range.getLeft())), -1, -1),
                    new RangeLimit(Cf.list(YTree.longNode(range.getRight())), -1, -1)
            );
        }
        yt.tables().read(path, YTableEntryTypes.YSON, consumer);
    }

    /**
     * Читает поле field YSON-структуры из строчки row, если структура хранится в колонке column
     *
     * @param row    Строка, прочитанная из YT
     * @param column Название колонки, где лежит структура
     * @param field  Поле в структуре
     * @return Прочитанное значение поля структуры или null, если не найдена колонка, или её значение null, или
     * внутри структуры нет такого поля
     */
    @Nullable
    public static String getYsonField(YTreeMapNode row, String column, String field) {
        Optional<YTreeNode> columnValue = row.get(column);
        if (columnValue.isEmpty() || !columnValue.get().isMapNode()) {
            return null;
        }
        Optional<YTreeNode> fieldValue = columnValue.get().mapNode().get(field);
        return fieldValue.map(YTreeNode::stringValue).orElse(null);
    }
}
