package ru.yandex.direct.ytwrapper.chooser;

import java.time.Duration;
import java.util.Collection;
import java.util.Optional;
import java.util.concurrent.ThreadLocalRandom;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.PreDestroy;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.env.EnvironmentType;
import ru.yandex.direct.utils.InterruptedRuntimeException;
import ru.yandex.direct.ytwrapper.client.YtClusterConfig;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.locks.CypressLockProvider;

/**
 * Определение режима работы через блокировки в YT.
 */
@ParametersAreNonnullByDefault
public class WorkModeChooser<T> {
    private static final Logger logger = LoggerFactory.getLogger(WorkModeChooser.class);

    // Время между успешным пингом и следующим
    // важно: время лока транзакции в 2 раза больше
    private static final Duration LONG_PING_INTERVAL = Duration.ofSeconds(60);
    // Время между неуспешным пингом и следующим
    // Не имеет значения, при любом неудачном пинге - завершаем программу
    private static final Duration SHORT_PING_INTERVAL = Duration.ZERO;


    private static final Duration MIN_AWAIT_BETWEEN_WORK_MODE_LOCK_ATTEMPTS = Duration.ofSeconds(5);
    private static final Duration MAX_AWAIT_BETWEEN_WORK_MODE_LOCK_ATTEMPTS = Duration.ofSeconds(10);

    private final AppDestroyer appDestroyer;
    private final CypressLockProvider cypressLockProvider;

    private volatile boolean destroying;

    CypressLockProvider.LockImpl currentLock;
    private T currentWorkMode;
    private final Collection<T> workModes;

    public WorkModeChooser(
            YtClusterConfig lockingYtConfig,
            String workModeLocksSubpath,
            EnvironmentType environmentType,
            AppDestroyer appDestroyer,
            Collection<T> workModes
    ) {
        this.appDestroyer = appDestroyer;
        this.workModes = workModes;

        var workModeLockRootPath = YPath.simple(lockingYtConfig.getHome())
                .child(workModeLocksSubpath)
                .child(environmentType.name().toLowerCase());
        cypressLockProvider = new CypressLockProvider(lockingYtConfig.getProxy(), lockingYtConfig.getToken(),
                Optional.of(lockingYtConfig.getUser()), workModeLockRootPath, LONG_PING_INTERVAL, SHORT_PING_INTERVAL);
        logger.info("Initialized WorkModeChooser. Lock directory is {}", workModeLockRootPath);
    }

    /**
     * Может бросить {@link IllegalStateException} если сеть прервалась и приложение уничтожается.
     */
    @Nonnull
    public synchronized T getOrRequestWorkMode() {
        if (appDestroyer.hasError() || destroying) {
            throw new IllegalStateException("Can not get work mode. App is destroying.", appDestroyer.getError());
        }
        if (currentWorkMode != null) {
            return currentWorkMode;
        }
        return requestWorkModeFromYt();
    }

    private T requestWorkModeFromYt() {
        while (currentWorkMode == null) {
            for (T workMode : workModes) {
                // в данном контексте, так как AppDestoyer не лежит в Completer, он совсем не срочный,
                // и вполне может подождать таймаута между проверками
                if (appDestroyer.hasError() || destroying) {
                    throw new IllegalStateException("Can not get work mode. App is destroying.",
                            appDestroyer.getError());
                }

                logger.info("Trying to take lock for workMode {}", workMode);
                var lock = tryToTakeLock(workMode);
                if (lock != null) {
                    logger.info("Got lock for workMode {}", workMode);
                    currentWorkMode = workMode;
                    currentLock = lock;
                    return workMode;
                }
            }
            Duration await = Duration.ofMillis(ThreadLocalRandom.current().nextLong(
                    MIN_AWAIT_BETWEEN_WORK_MODE_LOCK_ATTEMPTS.toMillis(),
                    MAX_AWAIT_BETWEEN_WORK_MODE_LOCK_ATTEMPTS.toMillis() + 1));
            logger.info("Failed to get locks for all work modes. Will try again after {}", await);
            try {
                Thread.sleep(await.toMillis());
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new InterruptedRuntimeException();
            }
        }
        return currentWorkMode;
    }

    @Nullable
    private CypressLockProvider.LockImpl tryToTakeLock(T workMode) {
        var lockName = workMode + "-lock";
        // если кластер yt_local и падает здесь
        // Для запуска на дев-сервере или разработческом ноутбуке:
        // Выполнить всё по инструкции https://github.yandex-team.ru/yt/docker.git
        try {
            var candidateLock = cypressLockProvider.createLock(lockName, this::onTransactionPingError);
            if (candidateLock.tryLock()) {
                return candidateLock;
            }
        } catch (RuntimeException e) {
            logger.warn("Failed to take yt lock", e);
        }
        return null;
    }

    // предполагается что при вызове этого метода лок уже взят или поток остановлен.
    @PreDestroy
    public void destroy() {
        if (destroying) {
            throw new IllegalStateException("Already destroying!!");
        }
        destroying = true;

        var lock = currentLock;
        currentWorkMode = null;
        currentLock = null;
        if (lock != null && lock.isTaken()) {
            lock.unlock();
            logger.info("App is destroyed, workMode transaction aborted.");
        } else {
            logger.info("App is destroyed, workMode transaction was null.");
        }
    }

    private void onTransactionPingError(Exception exc) {
        logger.error("Can not ping transaction for work mode. Destroying.", exc);
        // если пропала сеть иключение подхватится при первом запуске Completer.
        appDestroyer.destroy(exc);
    }
}
