package ru.yandex.direct.ytwrapper.client;

import java.nio.file.Path;
import java.util.Optional;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.typesafe.config.Config;
import org.apache.commons.lang3.StringUtils;

import ru.yandex.direct.utils.io.FileUtils;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.yql.response.yt.AclDto;

/**
 * Реализация {@link YtClusterConfig} поверх Typesafe'ного {@link Config}.
 */
@ParametersAreNonnullByDefault
public class YtClusterTypesafeConfig implements YtClusterConfig {
    private final Config ytClusterConfig;
    private final AclDto defaultYqlAcl;
    private final YtCluster cluster;

    YtClusterTypesafeConfig(YtCluster cluster, Config ytClusterConfig) {
        this.cluster = cluster;
        this.ytClusterConfig = ytClusterConfig;
        this.defaultYqlAcl = createDefaultYqlAcl(ytClusterConfig);
    }

    /**
     * @return значение параметра {@code "proxy"} из конфигурации кластера
     */
    @Override
    public String getProxy() {
        return ytClusterConfig.getString("proxy");
    }

    /**
     * @return аутентификационный токен из параметра {@code "token"} конфигурации кластера
     * или из файла {@link #getTokenPath()}, если параметр {@code "token"} не задан
     */
    @Override
    public String getToken() {
        return findString("token")
                .orElseGet(() -> FileUtils.slurp(getTokenPath()).trim());
    }

    @Override
    public String getYqlToken() {
        return findString("yql_token")
                .orElseGet(() -> FileUtils.slurp(getYqlTokenPath()).trim());
    }

    @Nullable
    @Override
    public AclDto getDefaultYqlAcl() {
        return defaultYqlAcl;
    }

    /**
     * @return значение параметра {@code "token_file"} из конфигурации кластера
     * или {@value #DEFAULT_YT_TOKEN_PATH}
     */
    private Path getTokenPath() {
        String tokenPath = findString("token_file")
                .orElse(DEFAULT_YT_TOKEN_PATH);
        return FileUtils.expandHome(tokenPath);
    }

    /**
     * @return значение параметра {@code "yql_token_file"} из конфигурации кластера
     * или {@value #DEFAULT_YQL_TOKEN_PATH}
     */
    private Path getYqlTokenPath() {
        String tokenPath = findString("yql_token_file")
                .orElse(DEFAULT_YQL_TOKEN_PATH);
        return FileUtils.expandHome(tokenPath);
    }

    /**
     * @return значение параметра {@code "home"} из конфигурации кластера
     */
    @Override
    public String getHome() {
        return ytClusterConfig.getString("home");
    }

    /**
     * @return значение параметра {@code "user"} из конфигурации кластера
     * или имя текущего пользователя из системного свойства {@code "user.name"}
     */
    @Override
    public String getUser() {
        return findString("user")
                .map(StringUtils::trimToNull)
                .orElseGet(() -> System.getProperty("user.name"));
    }

    @Override
    public YtCluster getCluster() {
        return cluster;
    }

    private Optional<String> findString(String path) {
        if (ytClusterConfig.hasPath(path)) {
            return Optional.of(ytClusterConfig.getString(path));
        }
        return Optional.empty();
    }

    @SuppressWarnings("SameParameterValue")
    private Optional<Integer> findInt(String path) {
        if (ytClusterConfig.hasPath(path)) {
            return Optional.of(ytClusterConfig.getInt(path));
        }
        return Optional.empty();
    }

    @Nullable
    private AclDto createDefaultYqlAcl(Config cfg) {
        if (!cfg.hasPath("default_yql_acl")) {
            return null;
        }
        var aclConfig = cfg.getConfig("default_yql_acl");
        return new AclDto(
                aclConfig.hasPath("can_read")
                        ? Set.copyOf(aclConfig.getStringList("can_read"))
                        : null,
                aclConfig.hasPath("can_write")
                        ? Set.copyOf(aclConfig.getStringList("can_write"))
                        : null,
                aclConfig.hasPath("can_private_execute")
                        ? Set.copyOf(aclConfig.getStringList("can_private_execute"))
                        : null
        );
    }
}
