package ru.yandex.direct.ytwrapper.client;

import java.util.List;
import java.util.concurrent.TimeoutException;
import java.util.function.Function;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.ytwrapper.exceptions.OperationRunningException;
import ru.yandex.direct.ytwrapper.model.YtCluster;

public class YtExecutionUtil {
    private static final Logger logger = LoggerFactory.getLogger(YtExecutionUtil.class);

    /**
     * Пытается инициализировать {@code initialize} и выполнить {@code doAction} некоторое действие на YT кластере.
     * Если при выполнении действия происходит ошибка, происходит попытка работы со следующим
     * кластером из {@code clustersByPriority} (если ошибка - {@link TimeoutException}, то стучимся на следующей кластер
     * только при отрицательном {@param breakIfTimeout})
     */
    public static <C, T> T executeWithFallback(List<YtCluster> clustersByPriority, Function<YtCluster, C> initialize,
                                               Function<C, T> doAction, boolean breakIfTimeout) {
        if (clustersByPriority.isEmpty()) {
            throw new YtExecutionException("No clusters available");
        }
        RuntimeException lastException = null;
        for (YtCluster cluster : clustersByPriority) {
            C initialized = initialize.apply(cluster);
            try {
                return doAction.apply(initialized);
            } catch (RuntimeException e) {
                logger.info("Error during YT action call", e);
                lastException = e;
                if (e instanceof OperationRunningException) {
                    Throwable cause = e.getCause();
                    if ((breakIfTimeout && cause instanceof TimeoutException)
                            || cause instanceof InterruptedException) {
                        // Если cause ошибка -- Timeout или прерывание, не делаем перезапрос на соседний кластер
                        break;
                    }
                }
            }
        }
        throw new YtExecutionException("Can't get result of YT of action", lastException);
    }

    public static <C, T> T executeWithFallback(List<YtCluster> clustersByPriority, Function<YtCluster, C> initialize,
                                               Function<C, T> doAction) {
        return executeWithFallback(clustersByPriority, initialize, doAction, true);
    }
}
