package ru.yandex.direct.ytwrapper.client;

import java.time.Duration;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.PreDestroy;

import io.netty.channel.nio.NioEventLoopGroup;
import org.apache.commons.lang3.tuple.Pair;
import org.jooq.Table;

import ru.yandex.direct.ytwrapper.dynamic.YtDynamicConfig;
import ru.yandex.direct.ytwrapper.dynamic.YtQueryComposer;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.direct.ytwrapper.model.YtDynamicOperator;
import ru.yandex.direct.ytwrapper.model.YtOperator;
import ru.yandex.direct.ytwrapper.model.YtSQLSyntaxVersion;
import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.YtConfiguration;
import ru.yandex.inside.yt.kosher.impl.YtUtils;
import ru.yandex.inside.yt.kosher.impl.operations.jars.SingleUploadFromClassPathJarsProcessor;
import ru.yandex.inside.yt.kosher.impl.ytbuilder.YtSyncHttpBuilder;
import ru.yandex.misc.io.http.Timeout;
import ru.yandex.yql.YqlDataSource;
import ru.yandex.yql.settings.YqlProperties;
import ru.yandex.yt.ytclient.bus.BusConnector;
import ru.yandex.yt.ytclient.bus.DefaultBusConnector;

/**
 * Компонент для получения инстансов {@link YtOperator} по кластеру
 */
@ParametersAreNonnullByDefault
public class YtProvider {
    private static final YPath DEFAULT_TMP_DIR = YPath.simple("//tmp/direct-java-yt-wrapper");
    private static final YtSQLSyntaxVersion DEFAULT_YT_SQL_SYNTAX_VERSION = YtSQLSyntaxVersion.SQLv1;

    private final ConcurrentHashMap<YtCluster, Yt> kosherYtCache = new ConcurrentHashMap<>();
    private final ConcurrentHashMap<YtCluster, YtDynamicOperator> dynamicYtCache = new ConcurrentHashMap<>();
    private final ConcurrentHashMap<Pair<YtCluster, YtSQLSyntaxVersion>, YqlDataSource> yqlCache =
            new ConcurrentHashMap<>();
    private final YtClusterConfigProvider clusterToConfig;
    private final BusConnector connector;
    private final YtDynamicConfig dynConfig;
    private final YtQueryComposer ytQueryComposer;
    private volatile boolean isDynamicYtCacheClosed = false;
    private final ReadWriteLock dynamicYtCacheCloseLock = new ReentrantReadWriteLock();


    public YtProvider(YtClusterConfigProvider clusterToConfig,
                      YtDynamicConfig dynConfig, YtQueryComposer ytQueryComposer) {
        this.clusterToConfig = clusterToConfig;
        this.dynConfig = dynConfig;
        this.ytQueryComposer = ytQueryComposer;
        connector = new DefaultBusConnector(new NioEventLoopGroup(10), true);
    }

    /**
     * Получить конфигурацию заданного кластера
     */
    public YtClusterConfig getClusterConfig(YtCluster cluster) {
        return clusterToConfig.get(cluster);
    }

    public Yt get(YtCluster cluster) {
        return kosherYtCache.computeIfAbsent(cluster, this::createYt);
    }

    public YqlDataSource getYql(YtCluster cluster, YtSQLSyntaxVersion syntaxVersion) {
        Pair<YtCluster, YtSQLSyntaxVersion> key = Pair.of(cluster, syntaxVersion);
        return yqlCache.computeIfAbsent(key, this::createYql);
    }

    /**
     * Получить оператор для выполнения операции для заданного кластера
     *
     * @param cluster нужный кластер
     * @return инстанс {@link YtOperator}, смотрящий в выбранный кластер
     */
    public YtOperator getOperator(YtCluster cluster) {
        return getOperator(cluster, DEFAULT_YT_SQL_SYNTAX_VERSION);
    }

    /**
     * Получить оператор для выполнения операции для заданного кластера
     *
     * @param cluster       нужный кластер
     * @param syntaxVersion версия YQL синтаксиса
     * @return инстанс {@link YtOperator}, смотрящий в выбранный кластер
     */
    public YtOperator getOperator(YtCluster cluster, YtSQLSyntaxVersion syntaxVersion) {
        return new YtOperator(getClusterConfig(cluster), get(cluster), this, cluster, syntaxVersion);
    }

    /**
     * Получить оператор для выполнения операции для заданного кластера
     */
    public YtOperator getOperator(YtCluster cluster, Duration httpClientSocketTimeout,
                                  Duration httpClientConnectTimeout,
                                  int simpleCommandsRetries) {
        return new YtOperator(
                getClusterConfig(cluster),
                createYt(cluster, httpClientSocketTimeout, httpClientConnectTimeout, simpleCommandsRetries),
                this,
                cluster,
                DEFAULT_YT_SQL_SYNTAX_VERSION);
    }

    private Yt createYt(YtCluster cluster) {
        YtConfiguration ytConfiguration = createDefaultYtConfigurationBuilder(cluster)
                // todo mspirit to config
                .withSimpleCommandsRetries(4)
                .build();
        return createDefaultYtBuilder(ytConfiguration)
                .build();
    }

    private Yt createYt(YtCluster cluster, Duration httpClientSocketTimeout, Duration httpClientConnectTimeout,
                        int simpleCommandsRetries) {
        YtConfiguration ytConfiguration = createDefaultYtConfigurationBuilder(cluster)
                .withSimpleCommandsRetries(simpleCommandsRetries)
                .build();
        return createDefaultYtBuilder(ytConfiguration)
                .withSimpleCommandsClient(YtUtils.buildHttpClient(
                        new Timeout(httpClientSocketTimeout.toMillis(), httpClientConnectTimeout.toMillis()),
                        ytConfiguration.getHttpClientMaxConnectionCount()))
                .build();
    }

    private YtConfiguration.Builder createDefaultYtConfigurationBuilder(YtCluster cluster) {
        YtClusterConfig ytClusterConfig = clusterToConfig.get(cluster);
        String token = ytClusterConfig.getToken();

        YtConfiguration.Builder builder = YtConfiguration.builder();
        // Для клиента к локальному YT убираем доп опции вида scheduling_tag_filter="porto"
        if (cluster == YtCluster.YT_LOCAL) {
            builder.withSpecPatch(null);
        }
        return builder
                .withApiHost(ytClusterConfig.getProxy())
                .withToken(token)
                .withTmpDir(DEFAULT_TMP_DIR);
    }

    private YtSyncHttpBuilder createDefaultYtBuilder(YtConfiguration ytConfiguration) {
        return Yt.builder(ytConfiguration)
                .http()
                .withJarsProcessor(
                        new SingleUploadFromClassPathJarsProcessor(
                                DEFAULT_TMP_DIR.child("jars"),
                                YtUtils.DEFAULT_CACHE_DIR,
                                true
                        )
                );
    }

    private YqlDataSource createYql(Pair<YtCluster, YtSQLSyntaxVersion> key) {
        YtClusterConfig ytClusterConfig = clusterToConfig.get(key.getLeft());
        return new YqlDataSource(
                "jdbc:yql://yql.yandex.net:443/" + key.getLeft().getName()
                        + "?syntaxVersion=" + key.getRight().getVersionCode(),
                new YqlProperties().withCredentials("unused", ytClusterConfig.getYqlToken())
        );
    }

    /**
     * Получить оператор для выполнения операций с динамическими таблицами заданного кластера
     * Метод вызывается из разных потоков, поэтому нужна блокировка, чтобы не случилось так, что во время вычиления
     * нового опреатора, вызвался метод close(), закрывающий все существующие операторы, а новый вычисленный оператор
     * так и остался незакрытым
     * <p>
     * NB: Клиент возвращается сразу после создания, без ожидания его инициализации
     *
     * @param cluster нужный кластер
     * @return инстанс {@link YtDynamicOperator}, смотрящий в выбранный кластер
     */
    public YtDynamicOperator getDynamicOperatorUnsafe(YtCluster cluster) {
        dynamicYtCacheCloseLock.readLock().lock();
        try {
            if (isDynamicYtCacheClosed) {
                throw new IllegalStateException("Can't get dynamic operator due to closed YtProvider");
            }
            return dynamicYtCache.computeIfAbsent(cluster, this::createDynamicOperator);
        } finally {
            dynamicYtCacheCloseLock.readLock().unlock();
        }
    }

    /**
     * Получить оператор для выполнения операций с динамическими таблицами заданного кластера
     *
     * @param cluster нужный кластер
     * @return оптимистично инициализированный (ждём не дольше таймаута) инстанс {@link YtDynamicOperator} для
     * указанного кластера
     */
    public YtDynamicOperator getDynamicOperator(YtCluster cluster) {
        return getDynamicOperatorUnsafe(cluster).optimisticallyInitialized();
    }

    private YtDynamicOperator createDynamicOperator(YtCluster cluster) {
        YtClusterConfig config = clusterToConfig.get(cluster);
        return new YtDynamicOperator(cluster, config, dynConfig, ytQueryComposer, connector);
    }

    @PreDestroy
    public void close() {
        dynamicYtCacheCloseLock.writeLock().lock();
        try {
            isDynamicYtCacheClosed = true;
            dynamicYtCache.values().forEach(YtDynamicOperator::close);
        } finally {
            dynamicYtCacheCloseLock.writeLock().unlock();
            connector.close();
        }
    }

    public String getPathToTable(Table table) {
        return ytQueryComposer.getPathToTable(table);
    }
}
