package ru.yandex.direct.ytwrapper.dynamic;

import java.util.Map;
import java.util.function.Function;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.jooq.Select;
import org.jooq.Table;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import static java.util.regex.Pattern.MULTILINE;

/**
 * Преобразует {@link Select} в строчку с запросом в YT-синтаксисе
 */
@ParametersAreNonnullByDefault
public class YtQueryComposer implements Function<Select, String> {
    private static final Logger logger = LoggerFactory.getLogger(YtQueryComposer.class);

    private static final char SINGLE_QUOTE = '\'';
    private static final Pattern BAD_BRACKETS = Pattern.compile("ON\\s*\\(\\s*\\(");

    private static final String BAD_NOT_EQUAL_SYMBOL = "<>";
    private static final String RIGHT_NOT_EQUAL_SYMBOL = "!=";
    private static final String GROUP_BY_STATEMENT = "GROUP BY";

    private static final Pattern LIMIT_OFFSET_PATTERN = Pattern.compile("LIMIT\\s+(\\d+)\\s+OFFSET\\s+(\\d+)");

    private final TableMappings tableMappings;
    private final boolean debug;

    public YtQueryComposer(TableMappings tableMappings) {
        this(tableMappings, false);
    }

    public YtQueryComposer(TableMappings tableMappings, boolean debug) {
        this.tableMappings = tableMappings;
        this.debug = debug;
    }

    /**
     * Превратить запрос в jooq-формате в строковый запрос, который может принять YT:
     * <p>
     * - Убираем SELECT из начала
     * - заменяем имена таблиц из моков на настоящие пути в YT
     * - меняем LEFT OUTER JOIN на LEFT JOIN
     * - заменяем двойные одинарные кавычки на {@literal "\'"}
     * - убираем лишние пробелы и переносы строк
     * - заменяем знак неравенства '<>' на '!='
     * - меняем местами LIMIT и OFFSET
     * - добавляет WITH TOTALS к запросу, если {@param withTotals} = true
     */
    private String composeQuery(Select selectQuery, boolean withTotals) {
        String query = selectQuery.toString();

        // todo maxlog: во имя производительности стоит использовать StringModifier
        // Убираем SELECT из начала
        if (query.startsWith("SELECT")) {
            query = query.replaceAll("^SELECT", "").trim();
        }

        // заменяем имена таблиц из моков на настоящие пути в YT
        Map<Table<?>, String> tableToRealPath = this.tableMappings.getTableMappings();
        for (Map.Entry<Table<?>, String> entry : tableToRealPath.entrySet()) {
            query = query.replace(getTarget(entry.getKey()), String.format("[%s]", entry.getValue()));
        }

        // меняем LEFT OUTER JOIN на LEFT JOIN
        query = query.replace("LEFT OUTER JOIN", "LEFT JOIN");

        query = escapeDoubleSingleQuotes(query);
        query = removeDoubleBracketsInJoin(query);

        if (!debug) {
            // убираем лишние пробелы из начала строк
            query = Pattern.compile("^\\s+", MULTILINE).matcher(query).replaceAll("");
            // убираем переносы строк (иногда jooq добавляет пробелы перед переносом)
            query = query
                    .replace(" \n", " ")
                    .replace("\n", " ");
        }

        query = query.replaceAll(BAD_NOT_EQUAL_SYMBOL, RIGHT_NOT_EQUAL_SYMBOL);

        var matcher = LIMIT_OFFSET_PATTERN.matcher(query);
        while (matcher.find()) {
            var origin = matcher.group(0);
            var reordered = "OFFSET " + matcher.group(2) + " LIMIT " + matcher.group(1);
            query = query.replace(origin, reordered);
        }

        if (withTotals) {
            query = addWithTotalsToQuery(query);
        }

        return query;
    }

    String addWithTotalsToQuery(String query) {
        int groupByIndex = query.indexOf(GROUP_BY_STATEMENT);
        if (groupByIndex < 0) {
            logger.warn("A query was found without GROUP BY statement. " +
                            "You need to add GROUP BY statement to use WITH TOTALS",
                    new IllegalStateException("Query without GROUP BY statement"));
            return query;
        }

        String afterGroupByPartOfQuery = query.substring(groupByIndex + GROUP_BY_STATEMENT.length());
        String[] queryFields = afterGroupByPartOfQuery.split("\\r?\\n|\\s+");
        String lastGroupByField = StreamEx.of(queryFields)
                .map(String::trim)
                .findFirst(param -> !param.isEmpty() && !param.contains(","))
                .orElse(null);
        int nextStatementIndex = lastGroupByField == null
                ? query.length()
                : query.indexOf(lastGroupByField, groupByIndex) + lastGroupByField.length();

        return query.substring(0, nextStatementIndex) + " \nWITH TOTALS " + query.substring(nextStatementIndex);
    }

    /**
     * jooq добавляет лишние скобки вокруг условия джойна, надо их убрать, а то YT поломается от такого запроса
     * <p>
     * Ищем выражение типа "ON (("
     * Если находим, то удаляем последнюю открывающую скобку и соответствующую закрывающую
     * Повторяем процесс.
     */
    private static String removeDoubleBracketsInJoin(String query) {
        StringBuilder sb = new StringBuilder(query);
        while (true) {
            Matcher matcher = BAD_BRACKETS.matcher(sb);
            if (!matcher.find()) {
                break;
            }

            int count = 2;
            int start = matcher.end();
            for (int i = start; i < sb.length(); i++) {
                char c = sb.charAt(i);

                if (c == '(') {
                    count++;
                } else if (c == ')') {
                    count--;
                }

                if (count == 0) {
                    sb.deleteCharAt(i);
                    sb.deleteCharAt(start - 1);
                    break;
                }
            }

            if (count != 0) {
                throw new IllegalArgumentException("Wrong query: " + query);
            }
        }
        return sb.toString();
    }

    /**
     * Заменяет две одинарные кавычки в строковом литерале на экранированную одинарную кавычку: {@literal "\'"}
     * Строковый литерал – это последовательность символов, заключенная в кавычки.
     * Ожидается, что на вход подается корректная строка полученная от jOOQ.
     * Дополнительных проверок на корректность в методе нет
     * <p>
     * Пример: запрос вида {@literal text = "'ab'"} jOOQ преобразует в {@literal text = '''ab'''}
     * А это функция преобразует в {@literal text = '\'ab\''}
     *
     * @see <a href="https://github.com/jOOQ/jOOQ/issues/5873"></a>
     * @see Select#toString()
     */
    private static String escapeDoubleSingleQuotes(String s) {
        final int firstDoubledSingleQuoteIndex = s.indexOf("''");

        //Эта проверка позволяет избежать создания нового массива символов и прохода по нему
        if (firstDoubledSingleQuoteIndex != -1) {
            char[] chars = s.toCharArray();
            boolean isFindOpeningSingleQuote = false;

            int i = 0;
            while (i < chars.length) {
                if (chars[i] == SINGLE_QUOTE) {
                    //Если эта открывающая кавычка для литерала, то пропускаем её
                    if (!isFindOpeningSingleQuote) {
                        isFindOpeningSingleQuote = true;
                    } else if ((i + 1) < chars.length && chars[i + 1] == SINGLE_QUOTE) {
                        /*
                        Нашли две кавычки. Первую заменяем на обратный слеш
                        и увеличиваем счетчик, чтобы вторую кавучку пропустить при следующей итерации
                         */
                        chars[i] = '\\';
                        i++;
                    } else {
                        //Значит это закрывающая кавычка для литерала. Сбрасываем флаг
                        isFindOpeningSingleQuote = false;
                    }
                }

                i++;
            }

            return String.valueOf(chars);
        }

        return s;
    }

    /**
     * Для переданной таблицы {@code table} возвращает её qualified name в строковом представлении.
     * Удобно для поиска вхождения названия таблицы в запросе для осуществления подстановки.
     */
    private static String getTarget(Table table) {
        return String.format("%s.%s", table.getSchema().getName(), table.getName());
    }

    /**
     * Возвращает строку с запросом в YT-синтаксисе
     *
     * @see #composeQuery(Select, boolean)
     */
    @Override
    public String apply(Select select) {
        return composeQuery(select, false);
    }

    public String apply(Select select, boolean withTotalStats) {
        return composeQuery(select, withTotalStats);
    }

    public String getPathToTable(Table table) {
        return tableMappings.getTableMappings().get(table);
    }
}
