package ru.yandex.direct.ytwrapper.dynamic.context;

import java.math.BigDecimal;
import java.util.Optional;
import java.util.function.Supplier;

import javax.annotation.Nullable;

import org.jooq.Field;
import org.jooq.Table;
import org.jooq.TableRecord;
import org.jooq.types.ULong;

import ru.yandex.inside.yt.kosher.impl.ytree.YTreeNodeUtils;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTree;
import ru.yandex.inside.yt.kosher.impl.ytree.object.YTreeRowSerializer;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;
import ru.yandex.misc.lang.number.UnsignedLong;
import ru.yandex.type_info.TiType;
import ru.yandex.yson.YsonConsumer;

public class YTableRecordSerializer<T extends TableRecord> implements YTreeRowSerializer<T> {

    private final Table<T> table;
    private final Supplier<T> recordSupplier;

    public YTableRecordSerializer(Table<T> table, Supplier<T> recordSupplier) {
        this.table = table;
        this.recordSupplier = recordSupplier;
    }

    @Override
    public void serializeRow(T obj, YsonConsumer consumer, boolean keyFieldsOnly, T compareWith) {
        serialize(obj, consumer);
    }

    @Override
    public void serialize(T obj, YsonConsumer consumer) {
        consumer.onBeginMap();
        for(var keyColumn : table.fields()) {
            var value = obj.get(keyColumn);
            if (value != null) {
                if (value instanceof ULong) {
                    value = UnsignedLong.valueOf(((ULong) value).longValue());
                }
                var valueNode = YTree.builder().value(value).build();
                consumer.onKeyedItem(keyColumn.getName());
                YTreeNodeUtils.walk(valueNode, consumer, true);
            }
        }
        consumer.onEndMap();
    }

    @Override
    public T deserialize(YTreeNode node) {
        var result = recordSupplier.get();
        var map = node.mapNode();
        for (Field column : table.fields()) {
            Optional<YTreeNode> valueNode = map.get(column.getName());
            if (valueNode.isPresent()) {
                var value = extractValue(column, valueNode.get());
                result.set(column, value);
                if (result instanceof TableRecordExt) {
                    ((TableRecordExt) result).setValueExt(column, valueNode.get());
                }
            }
        }
        return result;
    }

    @Override
    public TiType getColumnValueType() {
        return TiType.optional(TiType.yson());
    }

    private <T> Object extractValue(Field<T> field, @Nullable YTreeNode node) {
        if (node == null || node.isEntityNode()) {
            return null;
        }

        Class<T> type = field.getType();
        if (type.isAssignableFrom(Long.class)) {
            if (node.isBooleanNode()) {
                return node.boolValue() ? 1L : 0L;
            }
            return node.longValue();
        } else if (type.isAssignableFrom(Integer.class)) {
            return node.intValue();
        } else if (type.isAssignableFrom(ULong.class)) {
            return ULong.valueOf(node.longValue());
        } else if (type.isAssignableFrom(BigDecimal.class)) {
            if (node.isIntegerNode()) {
                return BigDecimal.valueOf(node.longValue());
            }
            return BigDecimal.valueOf(node.doubleValue());
        } else if (type.isAssignableFrom(Double.class)) {
            return node.doubleValue();
        } else if (type.isAssignableFrom(Float.class)) {
            return node.floatValue();
        } else if (type.isAssignableFrom(Boolean.class)) {
            return node.boolValue();
        } else if (type.isAssignableFrom(String.class)) {
            return node.stringValue();
        } else if (type.isEnum()) {
            @SuppressWarnings({"rawtypes", "unchecked"})
            Enum value = Enum.valueOf((Class<Enum>) type, node.stringValue());
            return value;
        }
        return null;
    }

    public interface TableRecordExt<R extends TableRecord<R>> extends TableRecord<R> {
        void setValueExt(Field column, YTreeNode yTreeNode);
    }
}
