package ru.yandex.direct.ytwrapper.dynamic.context;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import java.util.function.Supplier;

import org.jooq.Field;
import org.jooq.Select;
import org.jooq.Table;
import org.jooq.TableRecord;

import ru.yandex.direct.ytwrapper.client.YtExecutionUtil;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.direct.ytwrapper.model.YtField;
import ru.yandex.direct.ytwrapper.model.YtTable;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.tables.types.YTreeObjectEntryType;
import ru.yandex.inside.yt.kosher.ytree.YTreeMapNode;
import ru.yandex.yt.ytclient.proxy.AbstractLookupRowsRequest;
import ru.yandex.yt.ytclient.wire.UnversionedRowset;

import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * {@link YtDynamicContext} используется в связке с {@link YtDynamicContextProvider}.
 * Умеет делать fallback в случае ошибки выполнения запроса.
 */
public class YtDynamicContext {
    private final YtProvider ytProvider;
    private final List<YtCluster> clustersByPriority;
    private final Duration timeout;
    private final Integer maxSubqueries;

    public YtDynamicContext(YtProvider ytProvider, List<YtCluster> clustersByPriority, Duration timeout) {
        this(ytProvider, clustersByPriority, timeout, null);
    }

    public YtDynamicContext(YtProvider ytProvider, List<YtCluster> clustersByPriority, Duration timeout,
                            Integer maxSubqueries) {
        this.ytProvider = ytProvider;
        this.clustersByPriority = clustersByPriority;
        this.timeout = timeout;
        this.maxSubqueries = maxSubqueries;
    }

    public  <T extends TableRecord> List<T> lookupRows(List<T> keys, Supplier<T> recordSupplier) {
        if (keys.isEmpty()) {
            return Collections.emptyList();
        }
        var table = keys.get(0).getTable();
        var path = YPath.simple(ytProvider.getPathToTable(table));
        var entryType = new YTreeObjectEntryType(new YTableRecordSerializer(table, recordSupplier));
        return YtExecutionUtil.executeWithFallback(clustersByPriority,
                cluster -> ytProvider.get(cluster).tables(),
                ytTables -> {
                    List<T> result = new ArrayList(keys.size());
                    ytTables.lookupRows(path, entryType, keys, entryType, (Consumer<T>) result::add);
                    return result;
                },
                false
        );
    }

    /**
     * Считывает из таблицы строки (только с запрошенными полями) с помощью метода read_table через RPC-клиент.
     */
    public Collection<YTreeMapNode> readTable(Table<?> table, Collection<Field<?>> fields) {
        YtTable ytTable = new YtTable(getPathToTable(table.asTable()));
        return YtExecutionUtil.executeWithFallback(clustersByPriority,
                ytProvider::getOperator,
                operator -> operator.readTable(ytTable, mapList(fields,
                        field -> new YtField<>(field.getName(), field.getType())))
        );
    }

    /**
     * lookupRows в динтабличку через RPC-клиент. с перезапросами при таймауте
     */
    public UnversionedRowset executeLookup(AbstractLookupRowsRequest<?> request) {
        return YtExecutionUtil.executeWithFallback(clustersByPriority,
                ytProvider::getDynamicOperator,
                op -> op.runRpcCommandWithTimeout(() -> op.getYtClient().lookupRows(request)),
                false
        );
    }

    // proxy
    public String getPathToTable(Table table) {
        return ytProvider.getPathToTable(table);
    }

    /**
     * Возвращает результат выполнения запроса {@code select}
     */
    private UnversionedRowset executeSelect(Select select, boolean withTotalStats, Boolean breakIfTimeout) {
        return YtExecutionUtil.executeWithFallback(clustersByPriority,
                ytProvider::getDynamicOperator,
                dynOperator -> dynOperator.selectRows(select, withTotalStats, timeout, maxSubqueries),
                breakIfTimeout
        );
    }

    public UnversionedRowset executeSelect(Select select) {
        return executeSelect(select, false, true);
    }

    public UnversionedRowset executeSelect(Select select, boolean withTotalStats) {
        return executeSelect(select, withTotalStats, true);
    }

    public UnversionedRowset executeTimeoutSafeSelect(Select select) {
        return executeSelect(select, false, false);
    }

    public List<YtCluster> getClustersByPriority() {
        return new ArrayList<>(clustersByPriority);
    }
}
