package ru.yandex.direct.ytwrapper.dynamic.dsl;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import com.google.common.collect.Lists;
import org.apache.commons.lang3.StringUtils;
import org.jooq.Condition;
import org.jooq.Context;
import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.SQLDialect;
import org.jooq.conf.RenderKeywordStyle;
import org.jooq.conf.RenderNameStyle;
import org.jooq.conf.Settings;
import org.jooq.conf.StatementType;
import org.jooq.impl.CustomCondition;
import org.jooq.impl.CustomField;
import org.jooq.impl.DSL;
import org.jooq.impl.DefaultDataType;
import org.jooq.types.ULong;

import static ru.yandex.direct.utils.DateTimeUtils.MSK;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

public class YtDSL {
    private YtDSL() {
    }

    /**
     * Получить DSL-контекст для построения запроса в формате YT
     */
    public static DSLContext ytContext() {
        Settings settings = new Settings()
                .withRenderKeywordStyle(RenderKeywordStyle.UPPER)
                .withRenderNameStyle(RenderNameStyle.AS_IS)
                .withStatementType(StatementType.STATIC_STATEMENT);
        return DSL.using(SQLDialect.MYSQL, settings);
    }

    /**
     * Возвращает {@code epochSeconds} момента начала дня в таймзоне {@link ru.yandex.direct.utils.DateTimeUtils#MSK}.
     * <p>
     * Этим значением в таблицах со статистикой БК кодируются дни.
     */
    public static Field<Long> toEpochSecondsAtStartOfDate(LocalDate date) {
        long timestamp = date.atStartOfDay(MSK).toEpochSecond();
        return DSL.val(timestamp);
    }

    /**
     * {@code NULL} в YT можно использовать как значение любого типа без дополнительных приведений типов.
     *
     * <a href="https://yt.yandex-team.ru/docs/description/dynamic_tables/dyn_query_language#znachenie-null">Документация</a>
     */
    public static <T> Field<T> ytNull(Class<T> type) {
        return DSL.field("null", type);
    }

    /**
     * If-условие в формате YT
     */
    public static <T> Field<T> ytIf(Condition condition, Field<T> onTrue, Field<T> onFalse) {
        return DSL.field("IF({0}, {1}, {2})", onTrue.getType(), condition, onTrue, onFalse);
    }

    /**
     * Имитация {@code case-when-then-else}, реализованная через вложенные {@code if}'ы.
     * <p>
     * Поддерживает только прямое сравнение с целевым полем ({@code field.eq(case.when)}).
     */
    public static <T, R> Field<T> ytSwitch(Field<R> field, List<Case<R, T>> cases, T defaultValue) {
        Field<T> res = DSL.val(defaultValue);
        for (Case<R, T> caze : Lists.reverse(cases)) {
            res = ytIf(field.eq(caze.when), caze.then, res);
        }
        return res;
    }

    /**
     * IfNull-условие в формате YT
     */
    public static <T> Field<T> ytIfNull(Field<T> field, T valueIfFieldIsNull) {
        return ytIf(isNull(field), DSL.val(valueIfFieldIsNull), field);
    }

    /**
     * {@code true} в виде поля типа {@link Long}.
     * <p>
     * Сейчас {@code Boolean} колонки в jooq-моделях имеют тип {@link Long}.
     * Этот метод можно будет удалить, когда решим тикет DIRECT-75542.
     */
    public static Field<Long> ytTrue() {
        return DSL.field("true", Long.class);
    }

    /**
     * {@code false} в виде поля типа {@link Long}.
     * <p>
     * Сейчас {@code Boolean} колонки в jooq-моделях имеют тип {@link Long}.
     * Этот метод можно будет удалить, когда решим тикет DIRECT-75542.
     */
    public static Field<Long> ytFalse() {
        return DSL.field("false", Long.class);
    }

    /**
     * Кастинг любого числа в double в YT
     */
    public static <T extends Number> Field<BigDecimal> toDouble(Field<T> number) {
        return DSL.field("double({0})", BigDecimal.class, number);
    }

    /**
     * Кастинг любого числа в int64 в YT
     */
    public static <T extends Number> Field<Long> toLong(Field<T> number) {
        return DSL.field("int64({0})", Long.class, number);
    }

    /**
     * Кастинг любого числа в uint64 в YT
     */
    public static <T extends Number> Field<ULong> toULong(Field<T> longField) {
        return DSL.field("uint64({0})", ULong.class, longField);
    }

    /**
     * ВоULong в синтаксисе YT выглядит как число вида {@code 1234u}.
     *
     * <a href="https://wiki.yandex-team.ru/yt/userdoc/queries/#literaly">Документация YT на Вики</a>
     */
    public static Field<ULong> toULongLiteral(ULong value) {
        return DSL.field("{0}u", ULong.class, value);
    }

    /**
     * Функция для перевода значения поля в нижний регистр
     *
     * @param field строковое поле
     */
    public static Field<String> toLower(Field<String> field) {
        return DSL.field("lower({0})", String.class, field);
    }

    /**
     * Функция для перевода значения числового поля в строку
     *
     * @param field числовое поле
     * @return выражение функции по переданному полю
     */
    public static Field<String> numericToString(Field<Long> field) {
        return DSL.field("numeric_to_string({0})", String.class, field);
    }

    public static Field<Long> unixEpochStartOfWeek(Field<Long> field, Long zoneShift) {
        return DSL.field("timestamp_floor_week({0} + {1}) - {1}", Long.class, field, zoneShift);
    }

    public static Field<Long> unixEpochStartOfMonth(Field<Long> field, Long zoneShift) {
        return DSL.field("timestamp_floor_month({0} + {1}) - {1}", Long.class, field, zoneShift);
    }

    public static Field<Long> unixEpochStartOfQuarter(Field<Long> field, Long zoneShift) {
        return DSL.field("timestamp_floor_month(" +
                "timestamp_floor_month({0} + {1}) - " +
                "(((timestamp_floor_month({0} + {1}) - timestamp_floor_year({0} + {1})) / 2419200)) % 3 * 2419200" +
                ") - {1}", Long.class, field, zoneShift);
    }

    public static Field<Long> unixEpochStartOfYear(Field<Long> field, Long zoneShift) {
        return DSL.field("timestamp_floor_year({0} + {1}) - {1}", Long.class, field, zoneShift);
    }

    /**
     * Функция для передачи подготовленных данных в запрос
     * После вызова "transform" нужно через ".as('field')" указать алиас, чтобы поле можно было использовать в запросе
     * С полученным полем можно работать как обычно - используя сортировку, группировку, аггрегирование
     * <p>
     * Пример вызова:
     * YtDSL.transform("bid", ImmutableMap.of(1, 2, 3, 4)).as('forecast')
     * <p>
     * Получаем поле "transform(bid, (1, 3), (2, 4)) as forecast", где, например, 1, 3 - id баннеров
     * и 2, 4 - прогнозные клики из рекомендаций передаваемые в запрос
     *
     * @param field колонка-ключ для переданной мапы, например bid
     * @param map мапа значений
     */
    public static <K, V> Field<V> transform(Field<K> field, Map<K, V> map, Class<V> clazz) {
        // Сначала получаем список entries, чтобы гарантировать совпадение порядка ключей и значений
        ArrayList<Map.Entry<K, V>> entries = new ArrayList<>(map.entrySet());
        String keysString = StringUtils.join(mapList(entries, Map.Entry::getKey), ", ");
        String valuesString = StringUtils.join(mapList(entries, Map.Entry::getValue), ", ");

        return new CustomField<>("transform", DefaultDataType.getDataType(SQLDialect.MYSQL, clazz)) {
            @Override
            public void accept(Context<?> ctx) {
                ctx.sql("transform(").visit(field).sql(", (").sql(keysString).sql("), (").sql(valuesString).sql("))");
            }
        };
    }

    public static <T> Field<T> transform(Field<T> field, Map<T, T> map) {
        return transform(field, map, field.getType());
    }

    /**
     * Задание синонима - в YT возможно во всех секциях запроса и не только на верхнем уровне.
     *
     * <a href="https://yt.yandex-team.ru/docs/description/dynamic_tables/dyn_query_language#synonims">Документация</a>
     */
    public static <T> Field<T> alias(Field<T> field, Field<T> otherField) {
        return new CustomField<>(otherField.getUnqualifiedName(), field.getDataType()) {
            @Override
            public void accept(Context<?> ctx) {
                ctx.sql("(").visit(field).sql(" AS ").visit(otherField.getUnqualifiedName()).sql(")");
            }
        };
    }

    /**
     * Условие для проверки boolean поля, представленного у нас как Long, на истинность в YT
     */
    public static Condition isTrue(Field<Long> field) {
        return new CustomCondition() {
            @Override
            public void accept(Context<?> ctx) {
                ctx.visit(field);
            }
        };
    }

    /**
     * Условие для проверки поля на null в YT
     */
    public static Condition isNull(Field<?> field) {
        return new CustomCondition() {
            @Override
            public void accept(Context<?> ctx) {
                ctx.sql("is_null(").visit(field).sql(")");
            }
        };
    }

    /**
     * Условие для проверки поля вхождения подстроки в строку в YT
     *
     * @param substringCandidate подстрока
     * @param target             строка
     */
    public static Condition isSubstring(String substringCandidate, Field<String> target) {
        return isSubstring(DSL.val(substringCandidate), target);
    }

    /**
     * Условие для проверки поля вхождения подстроки в строку в YT
     *
     * @param substringCandidate поле с подстрокой
     * @param target             строка
     */
    public static Condition isSubstring(Field<String> substringCandidate, Field<String> target) {
        return new CustomCondition() {
            @Override
            public void accept(Context<?> ctx) {
                ctx.sql("is_substr(").visit(substringCandidate).sql(",").visit(target).sql(")");
            }
        };
    }

    public static Condition isSubstringIgnoringWhitespaceType(String substringCandidate, Field<String> target) {
        String unbreakableWhitespace = "\u00A0";
        String whitespace = "\u0020";
        return DSL.condition("is_substr({0},regex_replace_all({1},{2},{3}))",
                DSL.val(substringCandidate.replace(unbreakableWhitespace, whitespace)),
                DSL.val(unbreakableWhitespace),
                target,
                DSL.val(whitespace)
        );
    }

    /**
     * Агрегирующая функция first
     * Получить значение указанного в качестве аргумента выражения для одной из строк таблицы.
     * Не дает никаких гарантий о том, какая именно строка будет использована. Аналог функции any() в ClickHouse
     */
    public static <T> Field<T> first(Field<T> field) {
        return DSL.field("first({0})", field.getType(), field);
    }

    /**
     * Аггрегирующая функция SUM_IF
     */
    public static Field<BigDecimal> sumIf(Field<? extends Number> field, Condition condition) {
        return DSL.sum(YtDSL.ytIf(condition, field, null));
    }

    /**
     * Условие, заданное нативным YtQL-фрагментом
     */
    public static Condition nativeCondition(String nativeQueryCondition) {
        return new CustomCondition() {
            @Override
            public void accept(Context<?> ctx) {
                ctx.sql(nativeQueryCondition);
            }
        };
    }

    public static final class Case<T, R> {
        private final Field<T> when;
        private final Field<R> then;

        Case(Field<T> when, Field<R> then) {
            this.when = when;
            this.then = then;
        }

        public static <T, R> Case<T, R> of(Field<T> when, Field<R> then) {
            return new Case<>(when, then);
        }

        public static <T, R> Case<T, R> of(T when, Field<R> then) {
            return of(DSL.val(when), then);
        }
    }
}
