package ru.yandex.direct.ytwrapper.dynamic.dsl;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;
import java.util.Arrays;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.utils.DateTimeUtils;

import static org.apache.http.util.TextUtils.isEmpty;

/**
 * Утиль для парсинга результатов запросов к YT
 */
@ParametersAreNonnullByDefault
public class YtMappingUtils {
    /**
     * Используем умножение на 10**6, чтобы не терять важные для нас цифры после запятой.
     * YT не поддерживает длинную арифметику, а Double в этом плане ненадежен
     */
    public static final int YT_DECIMAL_SCALE = 6;
    /**
     * Масштаб для вычисления процентов.
     * Как {@link #YT_DECIMAL_SCALE}, только на два разряда больше
     */
    public static final int YT_PERCENT_DECIMAL_SCALE = YT_DECIMAL_SCALE + 2;
    private static final int INT_MONEY_SCALE = 2;
    private static final BigDecimal MONEY_SCALE = BigDecimal.valueOf(1L);

    private static final DateTimeFormatter MYSQL_DATE_TIME_FORMAT_OLD =
            DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss.S");

    private static final DateTimeFormatter MYSQL_DATE_TIME_FORMAT =
            DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");

    private static final String TRUE = "yes";

    private YtMappingUtils() {
    }

    public static <E extends Enum<E>> E enumFromString(@Nullable String value, Class<E> enumCls) {
        return !isEmpty(value) ? Enum.valueOf(enumCls, value.toUpperCase()) : null;
    }

    public static Boolean booleanFromYesNo(@Nullable String value) {
        return value != null ? value.equalsIgnoreCase(TRUE) : null;
    }

    public static LocalDateTime dateTimeFromString(@Nullable String value) {
        // TODO : можно убрать поддержку старого формата после выкатки релиза DIRECT-107126 в прод
        if (value != null) {
            try {
                return LocalDateTime.parse(value, MYSQL_DATE_TIME_FORMAT);
            } catch (DateTimeParseException e) {
                return LocalDateTime.parse(value, MYSQL_DATE_TIME_FORMAT_OLD);
            }
        } else {
            return null;
        }
    }

    public static Boolean booleanFromLong(@Nullable Long value) {
        return value != null ? value != 0 : null;
    }

    public static Set<String> stringToSet(@Nullable String value) {
        return value != null ? Arrays.stream(value.split(",")).collect(Collectors.toSet()) : null;
    }

    /**
     * Преобразовывает десятичное дробное число, представленное в виде микродолей {@code micros}, в {@link BigDecimal}.
     * <p>
     * В транспорте с Торгами цены представлены в виде микродолей. Это позволяет не терять точность при сериализациях.
     */
    public static BigDecimal fromMicros(@Nullable Long micros) {
        return micros != null ? scaleMoney(BigDecimal.valueOf(micros, YT_DECIMAL_SCALE).multiply(MONEY_SCALE)) : null;
    }

   public static BigDecimal fromMicros(@Nullable BigDecimal micros) {
        return micros != null ? scaleMoney(micros.scaleByPowerOfTen(-YT_DECIMAL_SCALE).multiply(MONEY_SCALE)) : null;
   }

    public static BigDecimal scaleMoney(BigDecimal money) {
        return money.setScale(INT_MONEY_SCALE, RoundingMode.HALF_UP);
    }

    public static LocalDate fromEpochSecond(@Nullable Long epochSecond) {
        return epochSecond != null ? LocalDate.ofInstant(Instant.ofEpochSecond(epochSecond), DateTimeUtils.MSK) : null;
    }
}
