package ru.yandex.direct.ytwrapper.dynamic.dsl;

import java.math.BigDecimal;
import java.time.LocalDate;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.inside.yt.kosher.ytree.YTreeMapNode;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;

/**
 * Утиль для построения запросов к YT
 */
@ParametersAreNonnullByDefault
public class YtQueryUtil {
    public static final BigDecimal DECIMAL_MULT = BigDecimal.TEN.pow(YtMappingUtils.YT_DECIMAL_SCALE);
    public static final BigDecimal PERCENT_DECIMAL_MULT = BigDecimal.TEN.pow(YtMappingUtils.YT_PERCENT_DECIMAL_SCALE);
    public static final BigDecimal CHIP_COST = BigDecimal.valueOf(30L);

    private YtQueryUtil() {
    }

    /**
     * Получить из ряда данных ответа YT процентные значения в формате БК
     *
     * @param row ряд данных
     * @param key ключ, по которому хранятся деньги
     */
    @Nullable
    public static BigDecimal extractBsPercent(YTreeMapNode row, String key) {
        YTreeNode node = row.get(key).orElse(null);
        if (node == null || node.isEntityNode()) {
            return null;
        }
        if (node.isIntegerNode()) {
            return YtMappingUtils.fromMicros(node.longValue());
        } else if (node.isDoubleNode()) {
            return YtMappingUtils.fromMicros((long) node.doubleValue());
        }
        throw new IllegalArgumentException("Cannot extract money from " + key + " = " + node);
    }

    /**
     * Получить из ряда данных ответа YT деньги в формате БК
     *
     * @param row ряд данных
     * @param key ключ, по которому хранятся деньги
     */
    @Nullable
    public static BigDecimal extractBsMoney(YTreeMapNode row, String key) {
        YTreeNode node = row.get(key).orElse(null);
        if (node == null || node.isEntityNode()) {
            return null;
        }
        if (node.isIntegerNode()) {
            return YtMappingUtils.fromMicros(node.longValue());
        } else if (node.isDoubleNode()) {
            return YtMappingUtils.fromMicros((long) node.doubleValue());
        }
        throw new IllegalArgumentException("Cannot extract money from " + key + " = " + node);
    }

    /**
     * Получить из лонговой YT-ноды BigDecimal, или null, если значение не задано
     *
     * @param row ряд данных
     * @param key ключ, по которому хранятся деньги
     */
    @Nullable
    public static BigDecimal longNodeBigDecimal(YTreeMapNode row, String key) {
        YTreeNode node = row.get(key).orElse(null);
        if (node == null || node.isEntityNode()) {
            return null;
        }
        return BigDecimal.valueOf(node.longValue());
    }

    /**
     * Получить из лонговой YT-ноды LocalDate по Москве, или null, если значение не задано
     *
     * @param row ряд данных
     * @param key ключ, по которому хранятся деньги
     */
    @Nullable
    public static LocalDate longNodeLocalDate(YTreeMapNode row, String key) {
        YTreeNode node = row.get(key).orElse(null);
        if (node == null || node.isEntityNode()) {
            return null;
        }
        return YtMappingUtils.fromEpochSecond(node.longValue());
    }
}
