package ru.yandex.direct.ytwrapper.dynamic.selector;

import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Timer;
import java.util.TimerTask;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;

import static org.apache.commons.lang3.Validate.notEmpty;
import static org.apache.commons.lang3.Validate.notNull;

/**
 * Выбирает кластер на основе сравнения атрибута указанной ноды Yt при помощи указанного компаратора.
 */
@ParametersAreNonnullByDefault
public class AttributeBasedClusterChooser implements ClusterChooser<Void> {
    private static final Logger logger = LoggerFactory.getLogger(AttributeBasedClusterChooser.class);

    public static final Comparator<YTreeNode> INSTANT_COMPARATOR =
            Comparator.comparing(a -> Instant.parse(a.stringValue()));
    public static final Comparator<YTreeNode> REVERSED_INSTANT_COMPARATOR = INSTANT_COMPARATOR.reversed();

    private final YtProvider ytProvider;
    private final Collection<YtCluster> clusters;
    private final YPath path;
    private final String attribute;
    private final Comparator<? super YTreeNode> attributeComparator;
    private final Timer timer;
    private final Duration interval;
    private volatile boolean isScheduled;
    private volatile List<YtCluster> clustersSorted;

    /**
     * Конструктор.
     *
     * @param timer               Таймер, на котором будет выполняться периодическое обновление информации о кластерах
     * @param interval            Интервал обновления информации о кластерах
     * @param ytProvider          Yt-провайдер
     * @param clusters            Список кластеров
     * @param path                Путь к ноде, для которой будет проверяться атрибут
     * @param attribute           Атрибут для проверки
     * @param attributeComparator Компаратор для выбора наиболее подходящего кластера в зависимости от атрибута
     */
    public AttributeBasedClusterChooser(Timer timer, Duration interval,
                                        YtProvider ytProvider, Collection<YtCluster> clusters, YPath path,
                                        String attribute, Comparator<? super YTreeNode> attributeComparator) {
        this.ytProvider = notNull(ytProvider);
        this.clusters = notEmpty(notNull(clusters));
        this.path = notNull(path);
        this.attribute = notEmpty(notNull(attribute)).startsWith("@")
                ? attribute.replaceFirst("@", "")
                : attribute;
        this.attributeComparator = notNull(attributeComparator);
        this.clustersSorted = new ArrayList<>();
        this.timer = timer;
        this.interval = interval;
        this.isScheduled = false;
    }

    @Nonnull
    @Override
    public Optional<YtCluster> getCluster(@Nullable Void param) {
        checkFirstRun();
        return clustersSorted.isEmpty() ? Optional.empty() : Optional.of(clustersSorted.get(0));
    }

    @Nonnull
    @Override
    public Optional<List<YtCluster>> getClustersOrdered(@Nullable Void param) {
        checkFirstRun();
        return Optional.of(clustersSorted);
    }

    private void checkFirstRun() {
        if (!isScheduled) {
            synchronized (this) {
                if (!isScheduled) {
                    chooseCluster();
                    timer.scheduleAtFixedRate(new UpdateClusterTimerTask(), interval.toMillis(), interval.toMillis());
                    isScheduled = true;
                }
            }
        }
    }

    /**
     * Сортирует кластеры по указанному атрибуту используя attributeComparator и запоминает
     * в {@link AttributeBasedClusterChooser#clustersSorted}.
     * <p>
     * В список попадают только те кластеры, где указанная нода/атрибут вообще есть
     */
    void chooseCluster() {
        clustersSorted = StreamEx.of(clusters)
                .mapToEntry(this::getAttribute)
                .nonNullValues()
                .sorted(Map.Entry.comparingByValue(attributeComparator))
                .keys()
                .toList();
        logger.debug("Clusters by attribute {} for path {}: {}", attribute, path, clustersSorted);
    }

    @Nullable
    private YTreeNode getAttribute(YtCluster ytCluster) {
        try {
            YPath yAttribute = path.attribute(attribute);
            return ytProvider.get(ytCluster).cypress().get(yAttribute);
        } catch (RuntimeException e) {
            logger.error("Error fetching attribute", e);
            return null;
        }
    }

    /**
     * Обёртка над {@link #chooseCluster()} для использования в {@link Timer#schedule(TimerTask, long, long)}
     */
    private class UpdateClusterTimerTask extends TimerTask {
        @Override
        public void run() {
            chooseCluster();
        }
    }
}
