package ru.yandex.direct.ytwrapper.dynamic.selector;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.ThreadLocalRandom;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Iterables;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.ytwrapper.model.YtCluster;

import static com.google.common.base.Preconditions.checkArgument;
import static java.util.Collections.emptyList;

/**
 * Позволяет выбирать кластер на основе весовой функции без параметров.
 */
public class WeightBasedClusterChooser implements ClusterChooser<Void> {
    private static final Logger logger = LoggerFactory.getLogger(WeightBasedClusterChooser.class);

    private final ClusterWeightFunction weightFunction;
    private final Collection<YtCluster> clusters;

    private volatile Map<ClusterFreshness, List<YtCluster>> clustersByFreshness;

    /**
     * С интервалом {@code interval} перевзвешивает кластеры из набора {@code clusters} функцией {@code weightFunction}.
     * Доступные кластеры возвращаются в методе {@link #getClustersOrdered()}.
     * Для балансировки доступные кластеры перемешиваются.
     */
    public WeightBasedClusterChooser(
            Timer timer,
            Duration interval,
            Collection<YtCluster> clusters,
            ClusterWeightFunction weightFunction) {
        this.clusters = clusters;
        this.clustersByFreshness = ImmutableMap.of(ClusterFreshness.FRESH, new ArrayList<>(clusters));
        this.weightFunction = weightFunction;
        // Используем #schedule(..), чтобы гарантировать не частоту исполнения task'и, а интервал между запусками
        timer.schedule(new UpdateClusterTimerTask(), 0, interval.toMillis());
    }

    /**
     * Обёртка над {@link #chooseCluster()} для использования в {@link Timer#schedule(TimerTask, long, long)}
     */
    private class UpdateClusterTimerTask extends TimerTask {
        @Override
        public void run() {
            chooseCluster();
        }
    }

    @Nonnull
    @Override
    public Optional<YtCluster> getCluster(@Nullable Void param) {
        return getClustersOrdered(null).map(list -> Iterables.getFirst(list, null));
    }

    @Nonnull
    @Override
    public Optional<List<YtCluster>> getClustersOrdered(@Nullable Void param) {
        checkArgument(param == null, "Can't operate with nonnull Void parameter");

        // Перемешаем свежие и не очень кластеры отдельно
        List<YtCluster> freshClusters = getShuffledClusters(ClusterFreshness.FRESH);
        List<YtCluster> rottenClusters = getShuffledClusters(ClusterFreshness.ROTTEN);

        // Затем соединим кластеры в единый список
        List<YtCluster> allClusters = new ArrayList<>(freshClusters);
        allClusters.addAll(rottenClusters);

        if (allClusters.isEmpty()) {
            return Optional.empty();
        } else {
            return Optional.of(allClusters);
        }
    }

    /**
     * Возвращает список кластеров заданной свежести ({@code freshness})
     */
    private List<YtCluster> getShuffledClusters(ClusterFreshness freshness) {
        // Так как clustersByFreshness может измениться в соседнем потоке, ограничиваемся одним чтением из него
        List<YtCluster> currentClusters = clustersByFreshness.get(freshness);
        if (currentClusters == null) {
            return emptyList();
        }
        List<YtCluster> ytClusters = new ArrayList<>(currentClusters);
        Collections.shuffle(ytClusters, ThreadLocalRandom.current());
        return ytClusters;
    }

    /**
     * Группируем кластеры по свежести и запоминаем в {@code clustersByFreshness}.
     */
    private void chooseCluster() {
        clustersByFreshness = StreamEx.of(clusters)
                .mapToEntry(this::failSafeWeightFunction)
                .invert()
                .grouping();
        logger.debug("Clusters by freshness: {}", clustersByFreshness);
    }

    /**
     * Обёртка над функцией взвешивания для кластера.
     * <p>
     * Не достучались до сервера или получили ошибку выполнения запроса -- возвращаем {@link ClusterFreshness#UNAVAILABLE}
     */
    @Nonnull
    private ClusterFreshness failSafeWeightFunction(YtCluster cluster) {
        try {
            return weightFunction.apply(cluster);
        } catch (RuntimeException e) {
            logger.warn("Can't get weight of cluster {}", cluster, e);
            return ClusterFreshness.UNAVAILABLE;
        }
    }

}
