package ru.yandex.direct.ytwrapper.model;

import java.time.Duration;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;
import ru.yandex.direct.utils.MonotonicTime;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;
import ru.yandex.yt.ytclient.proxy.ApiServiceTransaction;
import ru.yandex.yt.ytclient.proxy.request.ColumnFilter;
import ru.yandex.yt.ytclient.proxy.request.ExistsNode;
import ru.yandex.yt.ytclient.proxy.request.GetNode;
import ru.yandex.yt.ytclient.proxy.request.LockNode;
import ru.yandex.yt.ytclient.proxy.request.LockNodeResult;

import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.ytwrapper.YtUtils.STATE_ATTR;

public class LockWrapper extends TxWaitingWrapper {
    private static final Logger logger = LoggerFactory.getLogger(LockWrapper.class);


    private LockNodeResult lockNodeResult;

    LockWrapper(YtDynamicOperator ytDynamicOperator, ApiServiceTransaction tx) {
        super(ytDynamicOperator, tx);
    }

    @Override
    protected Logger getLogger() {
        return logger;
    }

    public LockWrapper createLock(LockNode request) {
        checkState(lockNodeResult == null, "lock already created");
        logger.info("Trying to acquire lock: {}", request.getArgumentsLogString());
        lockNodeResult = operator.runRpcCommandWithTimeout(tx::lockNode, request);
        logger.info("Lock id: {}", lockNodeResult.lockId);
        return this;
    }

    public boolean waitLock(Duration pollInterval, Duration timeout) {
        checkState(lockNodeResult != null, "lock was not created");

        var deadline = clock.getTime().plus(timeout);

        String path = "#" + lockNodeResult.lockId;
        ExistsNode existsNodeReq = new ExistsNode(path);
        GetNode getNodeReq = new GetNode(path)
                .setAttributes(ColumnFilter.of(STATE_ATTR));

        MonotonicTime iterationStart;
        while (!shutdown && (iterationStart = clock.getTime()).isBefore(deadline)) {
            logger.debug("Check state of lock {}", lockNodeResult.lockId);
            if (!operator.runRpcCommandWithTimeout(tx::existsNode, existsNodeReq)) {
                // лока нет совсем, например сдохла транзакция
                logger.warn("Lock {} was lost", lockNodeResult.lockId);
                return false;
            }

            YTreeNode node = operator.runRpcCommandWithTimeout(tx::getNode, getNodeReq);
            String state = node
                    .getAttributeOrThrow(STATE_ATTR)
                    .stringValue();
            if ("acquired".equals(state)) {
                logger.info("lock acquired");
                return true;
            }

            logger.debug("Still pending lock, Let's ping tx");
            pingTx();

            try (TraceProfile ignored = Trace.current().profile("LockWrapper:sleep")) {
                clock.sleep(pollInterval.minus(clock.getTime().minus(iterationStart)));
            } catch (InterruptedException e) {
                logger.info("Waiting for a lock interrupted");
                Thread.currentThread().interrupt();
            }
        }

        if (clock.getTime().isAtOrAfter(deadline)) {
            logger.warn("Lock wait timeout");
        }

        return false;
    }
}
