package ru.yandex.direct.ytwrapper.model;

import java.time.Duration;
import java.util.List;
import java.util.Optional;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;
import ru.yandex.direct.utils.MonotonicTime;
import ru.yandex.direct.ytwrapper.YtUtils;
import ru.yandex.inside.yt.kosher.common.GUID;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;
import ru.yandex.yt.ytclient.proxy.ApiServiceTransaction;
import ru.yandex.yt.ytclient.proxy.request.GetOperation;
import ru.yandex.yt.ytclient.proxy.request.StartOperation;

import static com.google.common.base.Preconditions.checkState;

public class OperationWrapper extends TxWaitingWrapper {
    private static final Logger logger = LoggerFactory.getLogger(OperationWrapper.class);

    private GUID guid;

    OperationWrapper(YtDynamicOperator ytDynamicOperator, ApiServiceTransaction tx) {
        super(ytDynamicOperator, tx);
    }

    @Override
    protected Logger getLogger() {
        return logger;
    }

    public OperationWrapper startOperation(StartOperation startOperation) {
        checkState(guid == null, "operation already started");
        logger.info("Start operation: {}", startOperation.getArgumentsLogString());

        guid = operator.runRpcCommandWithTimeout(tx::startOperation, startOperation);
        logOperationLink();
        return this;
    }

    public State waitOperation(Duration pollInterval, Duration timeout) {
        checkStarted();
        State lastState;

        MonotonicTime deadline = clock.getTime().plus(timeout);

        do {
            MonotonicTime iterationStart = clock.getTime();
            lastState = getOperation();
            lastState.logStatus();

            pingTx();

            if (lastState.isFinished()) {
                lastState.logResult();
                return lastState;
            }

            try (TraceProfile ignored = Trace.current().profile("OperationWrapper:sleep")) {
                clock.sleep(pollInterval.minus(clock.getTime().minus(iterationStart)));
            } catch (InterruptedException e) {
                logger.info("Waiting for an operation result interrupted");
                Thread.currentThread().interrupt();
            }
        } while (!shutdown && clock.getTime().isBefore(deadline));

        return lastState;
    }

    private State getOperation() {
        checkStarted();
        GetOperation getOperation = new GetOperation(guid)
                .setAttributes(State.getAttributes());

        YTreeNode yTreeNode = operator.runRpcCommandWithTimeout(operator.getYtClient()::getOperation, getOperation);
        return new State(yTreeNode);
    }

    public void logOperationLink() {
        checkStarted();
        logger.info("Operation started: https://yt.yandex-team.ru/{}/operations/{}",
                operator.getCluster().getName(), guid);
    }

    private void checkStarted() {
        checkState(guid != null, "operation was not started");
    }

    public static class State {
        private static final String ID_ATTR = "id";
        private static final String BRIEF_PROGRESS_ATTR = "brief_progress";
        private static final String RESULT_ATTR = "result";
        private static final String FINISH_TIME_ATTR = "finish_time";
        private static final String COMPLETED = "completed";
        private static final Set<String> finalStates = Set.of(COMPLETED, "failed", "aborted");
        private final String id;
        private final String state;
        private final String finishTime;
        private final YTreeNode briefProgress;
        private final YTreeNode result;

        private State(YTreeNode getOperationRes) {
            var map = getOperationRes.asMap();

            id = map.get(ID_ATTR).stringValue();
            state = map.get(YtUtils.STATE_ATTR).stringValue();
            finishTime = Optional.ofNullable(map.get(FINISH_TIME_ATTR)).map(YTreeNode::stringValue).orElse(null);
            briefProgress = Optional.ofNullable(map.get(BRIEF_PROGRESS_ATTR)).orElse(null);
            result = Optional.ofNullable(map.get(RESULT_ATTR)).orElse(null);
        }

        static List<String> getAttributes() {
            return List.of(ID_ATTR, YtUtils.STATE_ATTR, BRIEF_PROGRESS_ATTR, RESULT_ATTR, FINISH_TIME_ATTR);
        }

        public boolean isFinished() {
            return finalStates.contains(state);
        }

        public boolean isCompleted() {
            return COMPLETED.equals(state);
        }

        void logStatus() {
            logger.info("operation {} state: {}, progress: {}", id, state, briefProgress);
        }

        void logResult() {
            logger.info("operation {} finished at {} with result: {}", id, finishTime, result);
        }
    }
}
