package ru.yandex.direct.ytwrapper.model;

import java.time.Duration;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTree;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTreeBuilder;
import ru.yandex.inside.yt.kosher.ytree.YTreeMapNode;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;
import ru.yandex.yt.rpcproxy.EOperationType;
import ru.yandex.yt.ytclient.proxy.ApiServiceTransaction;
import ru.yandex.yt.ytclient.proxy.request.ColumnFilter;
import ru.yandex.yt.ytclient.proxy.request.GetNode;
import ru.yandex.yt.ytclient.proxy.request.LockMode;
import ru.yandex.yt.ytclient.proxy.request.LockNode;
import ru.yandex.yt.ytclient.proxy.request.StartOperation;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static ru.yandex.direct.ytwrapper.YtUtils.COMPRESSED_DATA_SIZE;
import static ru.yandex.direct.ytwrapper.YtUtils.COMPRESSION_RATIO_ATTR;
import static ru.yandex.direct.ytwrapper.YtUtils.DATA_SIZE_PER_JOB_ATTR;
import static ru.yandex.direct.ytwrapper.YtUtils.DATA_WEIGHT_ATTR;

public class TableMerger extends OperationWrapper {
    private static final Logger logger = LoggerFactory.getLogger(TableMerger.class);
    private static final long MAX_DATA_SIZE_PER_JOB = 32L * 1024L * 1024L * 1024L;
    private static final long MAX_ROW_WEIGHT = 128L * 1024L * 1024L;


    private final String table;
    private String pool;
    private double weight;
    private long desiredChunkSize;
    private boolean forceTransform;

    TableMerger(YtDynamicOperator ytDynamicOperator, ApiServiceTransaction tx, YPath table) {
        super(ytDynamicOperator, tx);
        this.table = table.justPath().toString();

        pool = null;
        weight = 1.0;
        forceTransform = false;
        desiredChunkSize = 512 * 1024L * 1024L;
    }

    private TableMerger(YtDynamicOperator ytDynamicOperator, ApiServiceTransaction tx, String table, String pool,
                        double weight, long desiredChunkSize, boolean forceTransform) {
        super(ytDynamicOperator, tx);
        this.table = table;
        this.pool = pool;
        this.weight = weight;
        this.desiredChunkSize = desiredChunkSize;
        this.forceTransform = forceTransform;
    }

    /**
     * Новый объект для выполнения мержа с настройками как у текущего.
     */
    public TableMerger newMerger() {
        return new TableMerger(operator, tx, table, pool, weight, desiredChunkSize, forceTransform);
    }

    public TableMerger withPool(String pool) {
        this.pool = checkNotNull(pool);
        return this;
    }

    public TableMerger withOperationWeight(double weight) {
        checkArgument(weight > 0);
        this.weight = weight;
        return this;
    }

    public TableMerger withForceTransform(boolean forceTransform) {
        this.forceTransform = forceTransform;
        return this;
    }

    public TableMerger withDesiredChunkSize(long desiredChunkSize) {
        checkArgument(desiredChunkSize > 0);
        this.desiredChunkSize = desiredChunkSize;
        return this;
    }

    /**
     * Запустить merge-операцию.
     * Можно использовать только один раз, для новой попытки с теми же настройками — используйте {@link #newMerger()}.
     *
     * @param pollInterval интервал опроса YT о завершении операции
     * @param timeout      время ожидания готовности операции
     * @return состояние операции на момент последнего опроса
     */
    public OperationWrapper.State doMerge(Duration pollInterval, Duration timeout) {
        YTreeMapNode spec = createMergeSpec();
        logger.debug("Merge specification: {}", spec);

        StartOperation startOperation = new StartOperation(EOperationType.OT_MERGE, spec);
        return startOperation(startOperation).waitOperation(pollInterval, timeout);
    }

    public LockNode createLockNodeRequest() {
        return new LockNode(table, LockMode.Exclusive).setWaitable(true);
    }

    private YTreeMapNode createMergeSpec() {
        GetNode req = new GetNode(table)
                .setAttributes(ColumnFilter.of(COMPRESSION_RATIO_ATTR, DATA_WEIGHT_ATTR, COMPRESSED_DATA_SIZE));
        YTreeNode node = operator.runRpcCommandWithTimeout(tx::getNode, req);

        Double compressionRatio = node.getAttribute(COMPRESSION_RATIO_ATTR).map(YTreeNode::doubleValue).orElse(null);
        Long dataWeight = node.getAttribute(DATA_WEIGHT_ATTR).map(YTreeNode::longValue).orElse(null);
        Long compressedDataSize = node.getAttribute(COMPRESSED_DATA_SIZE).map(YTreeNode::longValue).orElse(null);

        if (dataWeight != null && dataWeight > 0 && compressedDataSize != null) {
            compressionRatio = (double) compressedDataSize / dataWeight;
        }
        checkNotNull(compressionRatio, "compressionRation is unexpectedly null");

        long dataSizePerJob = Math.min(MAX_DATA_SIZE_PER_JOB, Math.max(1,
                (long) (desiredChunkSize / compressionRatio)));

        YTreeBuilder builder = YTree.mapBuilder()
                .key("mode").value("unordered")
                .key("combine_chunks").value(true)
                .key("force_transform").value(forceTransform)
                .key("input_table_paths")
                .beginList()
                .value(table)
                .endList()
                .key("output_table_path").value(table)
                .key("weight").value(weight)
                .key(DATA_SIZE_PER_JOB_ATTR).value(dataSizePerJob)
                .key("job_io")
                .beginMap()
                .key("table_writer")
                .beginMap()
                .key("desired_chunk_size").value(desiredChunkSize)
                .key("max_row_weight").value(MAX_ROW_WEIGHT)
                .endMap()
                .endMap()
                .key("unavailable_chunk_strategy").value("fail")
                .key("unavailable_chunk_tactics").value("fail");

        if (pool != null) {
            builder = builder.key("pool").value(pool);
        }

        return builder.buildMap();
    }
}
