package ru.yandex.direct.ytwrapper.model;

import java.time.Duration;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Supplier;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.Select;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.solomon.SolomonYtMonitorService;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;
import ru.yandex.direct.ytwrapper.client.YtClusterConfig;
import ru.yandex.direct.ytwrapper.dynamic.YtDynamicConfig;
import ru.yandex.direct.ytwrapper.dynamic.YtQueryComposer;
import ru.yandex.direct.ytwrapper.dynamic.logging.YtQueryLogRecord;
import ru.yandex.direct.ytwrapper.dynamic.logging.YtQueryLogger;
import ru.yandex.direct.ytwrapper.exceptions.OperationRunningException;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.yt.rpcproxy.ETransactionType;
import ru.yandex.yt.ytclient.bus.BusConnector;
import ru.yandex.yt.ytclient.proxy.ApiServiceTransaction;
import ru.yandex.yt.ytclient.proxy.ApiServiceTransactionOptions;
import ru.yandex.yt.ytclient.proxy.SelectRowsRequest;
import ru.yandex.yt.ytclient.proxy.YtClient;
import ru.yandex.yt.ytclient.proxy.request.RequestBase;
import ru.yandex.yt.ytclient.rpc.RpcCompression;
import ru.yandex.yt.ytclient.rpc.RpcCredentials;
import ru.yandex.yt.ytclient.rpc.RpcOptions;
import ru.yandex.yt.ytclient.wire.UnversionedRowset;

import static ru.yandex.direct.ytwrapper.YtUtils.createRpcProfile;

/**
 * Класс-помощник для выполнения основных операций с динамическими таблицами YT
 */
@ParametersAreNonnullByDefault
public class YtDynamicOperator {
    private static final Logger logger = LoggerFactory.getLogger(YtDynamicOperator.class);
    private static final Duration INITIALIZATION_TIMEOUT = Duration.ofSeconds(3);
    private static final Duration COMMAND_TIMEOUT = Duration.ofSeconds(20);
    private static final Duration PROXY_UPDATE_INTERVAL = Duration.ofMinutes(1); // умолчание в клиенте — 1 минута
    private static final SolomonYtMonitorService monitorService = new SolomonYtMonitorService();

    private final YtClientWrapper client;

    private final YtCluster cluster;
    private final YtDynamicConfig dynConfig;
    private final YtQueryComposer ytQueryComposer;
    private final CompletableFuture<YtDynamicOperator> initializedFuture;

    public YtDynamicOperator(YtCluster cluster, YtClusterConfig config,
                             YtDynamicConfig dynConfig, YtQueryComposer ytQueryComposer,
                             BusConnector busConnector) {
        this.cluster = cluster;
        this.dynConfig = dynConfig;
        this.ytQueryComposer = ytQueryComposer;

        client = createClient(busConnector, config);
        initializedFuture = wrap();
    }

    public static <T> T getWithTimeout(CompletableFuture<T> future, Duration timeout, String errorMessage) {
        Exception exc;
        try {
            return future.get(timeout.toMillis(), TimeUnit.MILLISECONDS);
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            exc = e;
        } catch (ExecutionException | TimeoutException e) {
            exc = e;
        }
        future.cancel(true);
        throw new OperationRunningException(errorMessage, exc);
    }

    public <REQ extends RequestBase<REQ>, RES> RES runRpcCommandWithTimeout(
            Function<REQ, CompletableFuture<RES>> command,
            REQ param
    ) {
        try (TraceProfile ignored = createRpcProfile(cluster)) {
            param.setTimeout(COMMAND_TIMEOUT);
            CompletableFuture<RES> result = command.apply(param);
            return getWithTimeout(result, COMMAND_TIMEOUT, "Command execution failed");
        }
    }

    public <RES> RES runRpcCommandWithTimeout(
            Supplier<CompletableFuture<RES>> command
    ) {
        try (TraceProfile ignored = createRpcProfile(cluster)) {
            CompletableFuture<RES> result = command.get();
            return getWithTimeout(result, COMMAND_TIMEOUT, "Command execution failed");
        }
    }

    private YtClientWrapper createClient(BusConnector busConnector, YtClusterConfig clusterConfig) {
        RpcOptions rpcOptions = new RpcOptions()
                .setGlobalTimeout(dynConfig.globalTimeout())
                .setFailoverTimeout(dynConfig.failoverTimeout())
                .setProxyUpdateTimeout(PROXY_UPDATE_INTERVAL)
                .setStreamingWriteTimeout(dynConfig.streamingWriteTimeout())
                .setStreamingReadTimeout(dynConfig.streamingReadTimeout())
                .setPingTimeout(Duration.ofSeconds(10));
        rpcOptions.setRpcClientSelectionTimeout(Duration.ofMillis(50));

        RpcCompression compression = new RpcCompression()
                .setRequestCodecId(dynConfig.requestCodecId())
                .setResponseCodecId(dynConfig.responseCodecId());

        var ytCluster = new ru.yandex.yt.ytclient.proxy.YtCluster(cluster.getName(),
                clusterConfig.getProxyHost(),
                clusterConfig.getProxyPort());

        var credentials = new RpcCredentials()
                .setUser(clusterConfig.getUser())
                .setToken(clusterConfig.getToken());

        try (TraceProfile ignored = Trace.current().profile("YtDynamicOperator:createClient")) {
            // FYI: иногда профайлинг оказывается за пределами трейсинга
            return new YtClientWrapper(busConnector, ytCluster, credentials, compression, rpcOptions);
        }
    }

    private CompletableFuture<YtDynamicOperator> wrap() {
        CompletableFuture<YtDynamicOperator> future = new CompletableFuture<>();
        future.completeOnTimeout(this, INITIALIZATION_TIMEOUT.toMillis(), TimeUnit.MILLISECONDS);

        client.waitProxies().whenComplete((r, ex) -> {
            if (ex != null) {
                logger.warn("Waiting proxies for {} failed", cluster, ex);
            } else {
                logger.info("Successfully waited proxies for {}", cluster);
            }
            future.complete(this);
        });

        return future;
    }

    /**
     * Сразу после создания YtClient не может выполнять запросы, потому что у него пустой список rpc-proxy.
     * Блокироваться на ожидании проксей прямо в конструкторе мы не хотим по ряду причин.
     * <p>
     * Позволяем подождать на стороне вызывающего кода:
     * таймаут {@link #INITIALIZATION_TIMEOUT} или время загрузке проксей, смотря что наступит раньше.
     * <p>
     * Если прокси не успели загрузиться за таймаут, клиент будет возвращен сразу (в нерабочем виде).
     * Обновление списка проксей клиент производит каждые {@link #PROXY_UPDATE_INTERVAL}.
     */
    public YtDynamicOperator optimisticallyInitialized() {
        return initializedFuture.join(); // IGNORE-BAD-JOIN DIRECT-149116
    }

    public void close() {
        logger.info("Shutting down dynamic operator for cluster {}", cluster);
        client.closeInternal();
    }

    /**
     * Вызывает {@link #runRpcCommand(YtClient, Function)} с локальным клиентом {@link #client}
     */
    public void runRpcCommand(Consumer<YtClient> consumer) {
        runRpcCommand(c -> {
            consumer.accept(c);
            return true;
        });
    }

    /**
     * Вызывает {@link #runRpcCommand(YtClient, Function)} с локальным клиентом {@link #client}
     */
    public <T> T runRpcCommand(Function<YtClient, T> function) {
        return runRpcCommand(client, function);
    }

    /**
     * Выполнить переданную функцию в транзакции.
     * Чтобы транзакция пинговалась, обязательно укажите опцию ping, иначе она завершится по таймауту.
     *
     * @param consumer  Функция, которая работает с данными. На вход ей поступит объект транзакции.
     * @param txOptions Опции транзакции.
     */
    public void runInTransaction(Consumer<ApiServiceTransaction> consumer, ApiServiceTransactionOptions txOptions) {
        runRpcCommand(
                c -> {
                    logger.info("Starting transaction");
                    try (ApiServiceTransaction transaction = startTransaction(c, txOptions)) {
                        logger.info("Transaction started: {} (timestamp={}, ping={}, sticky={})",
                                transaction.getId(),
                                transaction.getStartTimestamp(),
                                transaction.isPing(),
                                transaction.isSticky());

                        try {
                            consumer.accept(transaction);
                            logger.info("Committing transaction");
                            transaction.commit().join(); // IGNORE-BAD-JOIN DIRECT-149116
                            logger.info("Successfully committed");
                            return;
                        } catch (Exception e) {
                            logger.error("Got exception when executing payload or committing transaction", e);
                        }
                        logger.info("Going to roll back transaction");
                    }
                    throw new OperationRunningException("Transaction rolled back");
                }
        );
    }

    /**
     * Выполнить переданную функцию в транзакции.
     * Чтобы транзакция пинговалась, используйте аналогичный метод только с опциями.
     *
     * @param consumer функция, которая работает с данными. На вход ей поступит объект транзакции
     */
    public void runInTransaction(Consumer<ApiServiceTransaction> consumer) {
        runInTransaction(consumer, new ApiServiceTransactionOptions(ETransactionType.TT_MASTER).setSticky(true));
    }

    /**
     * Вызывает функцию для заданного {@link YtClient}
     */
    private <T> T runRpcCommand(YtClient client, Function<YtClient, T> function) {
        try (TraceProfile ignore = createRpcProfile(cluster)) {
            return function.apply(client);
        }
    }

    private ApiServiceTransaction startTransaction(YtClient client, ApiServiceTransactionOptions transactionOptions) {
        return client.startTransaction(transactionOptions).join(); // IGNORE-BAD-JOIN DIRECT-149116
    }

    public YtClient getYtClient() {
        return client;
    }

    /**
     * Возвращает кластер, для которого собран этот коннектор
     */
    public YtCluster getCluster() {
        return cluster;
    }

    public UnversionedRowset selectRows(Select query) {
        return selectRows(query, dynConfig.defaultSelectRowsTimeout());
    }

    public UnversionedRowset selectRows(Select query, Duration timeout) {
        String textQuery = ytQueryComposer.apply(query);
        return selectRows(textQuery, timeout);
    }

    public UnversionedRowset selectRows(Select query, Duration timeout, @Nullable Integer maxSubqueries) {
        return selectRows(query, false, timeout, maxSubqueries);
    }

    public UnversionedRowset selectRows(Select query, boolean withTotalStats,
                                        Duration timeout, @Nullable Integer maxSubqueries) {
        String textQuery = ytQueryComposer.apply(query, withTotalStats);
        return selectRows(textQuery, timeout, maxSubqueries);
    }

    public UnversionedRowset selectRows(String query) {
        return selectRows(query, dynConfig.defaultSelectRowsTimeout());
    }

    public UnversionedRowset selectRows(String query, Duration timeout) {
        return selectRows(query, timeout, null);
    }

    public UnversionedRowset selectRows(String query, Duration timeout, @Nullable Integer maxSubqueries) {
        if (timeout.compareTo(dynConfig.globalTimeout()) > 0) {
            logger.warn("Query timeout {} is greater than global {}", timeout, dynConfig.globalTimeout());
        }
        YtQueryLogRecord record = new YtQueryLogRecord(query);

        try {
            UnversionedRowset rowset = runRpcCommand(c -> {
                SelectRowsRequest request = SelectRowsRequest.of(query)
                        .setTimeout(timeout)
                        .setInputRowsLimit(dynConfig.inputRowsLimit())
                        .setOutputRowsLimit(dynConfig.outputRowsLimit());
                if (Objects.nonNull(maxSubqueries)) {
                    request.setMaxSubqueries(maxSubqueries);
                }
                return getWithTimeout(c.selectRows(request), timeout, "Error on executing selectRows");
            });
            record.setResultRows(rowset.getRows().size());
            return rowset;
        } catch (Exception e) {
            record.setException(e);
            throw e;
        } finally {
            YtQueryLogger.log(record);
            monitorService.sendMetrics(cluster.getName(), (long) (record.getEla() * 1000), record.getError() == null);
        }
    }

    public LockWrapper createLockWrapper(ApiServiceTransaction tx) {
        return new LockWrapper(this, tx);
    }

    public TableMerger createTableMerger(ApiServiceTransaction tx, YPath table) {
        return new TableMerger(this, tx, table);
    }

    private static class YtClientWrapper extends YtClient {
        private YtClientWrapper(
                BusConnector connector,
                ru.yandex.yt.ytclient.proxy.YtCluster ytCluster,
                RpcCredentials credentials,
                RpcCompression compression,
                RpcOptions options) {
            super(connector, List.of(ytCluster), ytCluster.getName(), null, credentials, compression, options);
        }

        @Override
        public void close() {
            // Защищаемся от закрытия клиента извне. Это нужно, так как клиент доступен потребителям во время
            // выполнения команд, и они могут его случайно попробовать закрыть. А клиент обычно общий
            throw new IllegalStateException("This client cannot be closed from outside");
        }

        private void closeInternal() {
            super.close();
        }
    }
}
