package ru.yandex.direct.ytwrapper.model;

import java.math.BigInteger;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Function;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;
import org.jooq.types.ULong;

import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTree;
import ru.yandex.inside.yt.kosher.ytree.YTreeEntityNode;
import ru.yandex.inside.yt.kosher.ytree.YTreeMapNode;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;

/**
 * Класс для представления поля в YT таблице
 *
 * @param <F> тип данных, хранящихся в поле. Поддерживаются типы assignable from ({@link Class#isAssignableFrom(Class)})
 *            Long, Integer, String, Double и Boolean
 */
@ParametersAreNonnullByDefault
public class YtField<F> {
    public enum SortOrder {
        DESCENDING,
        ASCENDING
    }

    private final String name;
    private final Class<F> valueClass;
    private final Function<YTreeNode, F> castFunction;
    private final SortOrder sortOrder;

    /**
     * @param name       название поля
     * @param valueClass класс типа данных, хранящихся в поле (типа F)
     */
    public YtField(String name, Class<F> valueClass) {
        this(name, valueClass, null);
    }

    /**
     * @param name       название поля
     * @param valueClass класс типа данных, хранящихся в поле (типа F)
     * @param sortOrder  если по полю есть сортировка, это ее направление; null, если сортировка не задана
     */
    public YtField(String name, Class<F> valueClass, @Nullable SortOrder sortOrder) {
        this.name = name;
        this.valueClass = valueClass;
        this.sortOrder = sortOrder;
        this.castFunction = makeCastFunction();
    }

    public static YtField<?> noTypeField(String name) {
        return new YtField<>(name, String.class);
    }

    /**
     * @return имя поля
     */
    public String getName() {
        return name;
    }

    /**
     * @return порядок сортировки или null, если по полю не сортируется
     */
    @Nullable
    public SortOrder getSortOrder() {
        return sortOrder;
    }

    /**
     * Извлечь из структуры данных, описывающей строку YT таблицы значение, хранящееся в данном поле.
     * Если такого поля в строке нет, или его значение равно null, возвращается defaultValue
     *
     * @param node         строка YT таблицы
     * @param defaultValue значение по умолчанию
     * @return значение, хранящееся в данном поле, или значение по умолчанию
     */
    @Nullable
    F extractValue(YTreeMapNode node, @Nullable F defaultValue) {
        Optional<YTreeNode> valOption = node.get(name);
        if (valOption.isPresent()) {
            YTreeNode val = valOption.get();
            if (val instanceof YTreeEntityNode) {
                return defaultValue;
            } else {
                return castFunction.apply(val);
            }
        } else {
            return defaultValue;
        }
    }

    @Nullable
    public F extractValue(YTreeMapNode node) {
        return extractValue(node, null);
    }

    /**
     * Вставить в структуру данных, описывающую строку YT таблицы значение
     *
     * @param node  строка YT таблицы
     * @param value значение для вставки умолчанию
     */
    public void insertValue(YTreeMapNode node, F value) {
        YTreeNode wrappedValue;
        if (valueClass.isAssignableFrom(Long.class)) {
            wrappedValue = YTree.integerNode((Long) value);
        } else if (valueClass.isAssignableFrom(Integer.class)) {
            wrappedValue = YTree.integerNode((Integer) value);
        } else if (valueClass.isAssignableFrom(String.class)) {
            wrappedValue = YTree.stringNode((String) value);
        } else {
            throw new IllegalArgumentException(String.format("Unsupported class: %s", valueClass));
        }
        node.put(name, wrappedValue);
    }

    @SuppressWarnings("unchecked")
    private Function<YTreeNode, F> makeCastFunction() {
        if (valueClass.isAssignableFrom(ULong.class)) {
            return val -> (F) ULong.valueOf(val.longValue());
        } else if (valueClass.isAssignableFrom(Long.class)) {
            return val -> (F) Long.valueOf(val.longValue());
        } else if (valueClass.isAssignableFrom(long.class)) {
            return val -> (F) Long.valueOf(val.longValue());
        } else if (valueClass.isAssignableFrom(Integer.class)) {
            return val -> (F) Integer.valueOf(val.intValue());
        } else if (valueClass.isAssignableFrom(int.class)) {
            return val -> (F) Integer.valueOf(val.intValue());
        } else if (valueClass.isAssignableFrom(String.class)) {
            return val -> (F) val.stringValue();
        } else if (valueClass.isAssignableFrom(Double.class) || valueClass.isAssignableFrom(double.class)) {
            return val -> (F) Double.valueOf(val.doubleValue());
        } else if (valueClass.isAssignableFrom(Boolean.class)) {
            return val -> (F) Boolean.valueOf(val.boolValue());
        } else if (valueClass.isAssignableFrom(BigInteger.class)) {
            return val -> (F) new BigInteger(Long.toUnsignedString(val.longValue()));
        } else if (valueClass.isAssignableFrom(YTreeNode.class)) {
            return val -> (F) val.cast();
        } else {
            throw new IllegalArgumentException(String.format("Unsupported class: %s", valueClass));
        }
    }

    public Map<String, String> getSchema() {
        ImmutableMap.Builder<String, String> builder = ImmutableMap.<String, String>builder()
                .put("name", name)
                .put("type", getType());
        if (sortOrder != null) {
            builder.put("sort_order", sortOrder.name().toLowerCase());
        }
        return builder.build();
    }

    private String getType() {
        if (valueClass.isAssignableFrom(Long.class)) {
            return "int64";
        } else if (valueClass.isAssignableFrom(String.class)) {
            return "string";
        } else if (valueClass.isAssignableFrom(Boolean.class)) {
            return "boolean";
        } else {
            throw new IllegalArgumentException(String.format("Unsupported class: %s", valueClass));
        }
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        YtField<?> ytField = (YtField<?>) o;
        return Objects.equals(name, ytField.name) &&
                Objects.equals(valueClass, ytField.valueClass);
    }

    @Override
    public int hashCode() {
        return Objects.hash(name, valueClass);
    }

    @Override
    public String toString() {
        return String.format("%s(\"%s\")", this.getClass().getSimpleName(), name);
    }
}
