package ru.yandex.direct.ytwrapper.model;

import java.sql.Connection;
import java.sql.SQLException;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.BiFunction;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Stopwatch;
import com.google.common.collect.Iterables;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.impl.SingletonSet;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.ytwrapper.YtUtils;
import ru.yandex.direct.ytwrapper.client.YtClusterConfig;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.exceptions.ReadException;
import ru.yandex.direct.ytwrapper.exceptions.RuntimeYqlException;
import ru.yandex.direct.ytwrapper.specs.OperationSpec;
import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.common.GUID;
import ru.yandex.inside.yt.kosher.cypress.RangeLimit;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.transactions.utils.YtTransactionsUtils;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTree;
import ru.yandex.inside.yt.kosher.tables.YTableEntryType;
import ru.yandex.inside.yt.kosher.tables.YTableEntryTypes;
import ru.yandex.inside.yt.kosher.transactions.Transaction;
import ru.yandex.inside.yt.kosher.ytree.YTreeListNode;
import ru.yandex.inside.yt.kosher.ytree.YTreeMapNode;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;
import ru.yandex.yql.ResultSetFuture;
import ru.yandex.yql.YqlDataSource;
import ru.yandex.yql.YqlPreparedStatement;

import static ru.yandex.direct.utils.FunctionalUtils.listToMap;

/**
 * Класс-помошник для выполнения основных операций в YT
 * <p>
 * ВНИМАНИЕ!!!
 * Методы selectRows работают только с динамическими таблицами.
 * Для статических таблиц используйте методы readTableXXXX.
 */
@ParametersAreNonnullByDefault
public class YtOperator {
    private static final Logger logger = LoggerFactory.getLogger(YtOperator.class);

    private static final String UPLOAD_TIME_ATTRIBUTE = "upload_time";
    private static final String MODIFICATION_TIME_ATTRIBUTE = "modification_time";
    private static final String ROW_COUNT_ATTRIBUTE = "row_count";

    /**
     * Настройки таймаутов для чтения снепшота таблицы.
     * Если понадобится менять - вытащить в конфигурацию или параметр метода.
     */
    private static final Duration SNAPSHOT_TRANSACTION_TIMEOUT = Duration.ofMinutes(5);
    private static final Duration SNAPSHOT_TRANSACTION_PING_PERIOD = Duration.ofSeconds(10);
    private static final String YQL_OPERATION_URL_BASE = "https://yql.yandex-team.ru/Operations/";

    private final YtClusterConfig ytClusterConfig;
    private final Yt yt;
    private final YtProvider ytProvider;
    private final YtCluster cluster;
    private final YtSQLSyntaxVersion ytSQLSyntaxVersion;

    public YtOperator(
            YtClusterConfig ytClusterConfig, Yt yt, YtProvider ytProvider, YtCluster cluster,
            YtSQLSyntaxVersion ytSQLSyntaxVersion
    ) {
        this.ytClusterConfig = ytClusterConfig;
        this.yt = yt;
        this.ytProvider = ytProvider;
        this.cluster = cluster;
        this.ytSQLSyntaxVersion = ytSQLSyntaxVersion;
    }

    public Yt getYt() {
        return yt;
    }

    public YtCluster getCluster() {
        return cluster;
    }

    /**
     * Запустить операцию по ее описанию
     *
     * @param operationSpec спека операции, которую нужно выполнить
     */
    public void runOperation(OperationSpec operationSpec) {
        operationSpec.run(yt);
    }

    /**
     * Получить строку с временем загрузки таблицы
     */
    public String readTableUploadTime(YtTable table) {
        return readTableStringAttribute(table, UPLOAD_TIME_ATTRIBUTE);
    }

    /**
     * Получить строку с временем загрузки таблицы
     */
    public Optional<String> tryReadTableUploadTime(YtTable table) {
        return tryReadTableStringAttribute(table, UPLOAD_TIME_ATTRIBUTE);
    }

    public void writeTableUploadTime(YtTable table, String uploadTime) {
        writeTableStringAttribute(table, UPLOAD_TIME_ATTRIBUTE, uploadTime);
    }

    /**
     * Получить строку с временем модификации таблицы
     */
    public String readTableModificationTime(YtTable table) {
        return readTableStringAttribute(table, MODIFICATION_TIME_ATTRIBUTE);
    }

    /**
     * Получить количество строк в таблице
     */
    public Long readTableRowCount(YtTable table) {
        return readTableNumericAttribute(table, ROW_COUNT_ATTRIBUTE);
    }

    /**
     * Получить строковый атрибут таблицы
     */
    public String readTableStringAttribute(YtTable table, String attributeName) {
        return readTableAttribute(table, attributeName).stringValue();
    }

    /**
     * Получить строковый атрибут таблицы
     */
    public Optional<String> tryReadTableStringAttribute(YtTable table, String attributeName) {
        return tryReadTableAttribute(table, attributeName)
                .map(YTreeNode::stringValue);
    }

    public void writeTableStringAttribute(YtTable table, String attributeName, String attributeValue) {
        writeTableStringAttribute(table.ypath(), attributeName, attributeValue);
    }

    /**
     * Записать строковый аттрибут таблицы
     *
     * @param tableYpath     ypath таблицы
     * @param attributeName  имя атрибута
     * @param attributeValue значение атрибута
     */
    public void writeTableStringAttribute(YPath tableYpath, String attributeName, String attributeValue) {
        YPath yPath = tableYpath.attribute(attributeName);
        yt.cypress().set(yPath, attributeValue);
    }

    /**
     * Получить числовой атрибут таблицы
     */
    public Long readTableNumericAttribute(YtTable table, String attributeName) {
        return readTableAttribute(table, attributeName).longValue();
    }

    YTreeNode readTableAttribute(YtTable table, String attributeName) {
        return tryReadTableAttribute(table, attributeName)
                .orElseThrow(() -> new RuntimeException(
                        String.format("missing attribute %s on %s", attributeName, table.getPath())));
    }

    private Optional<YTreeNode> tryReadTableAttribute(YtTable table, String attributeName) {
        return tryReadTableAttributes(table, Set.of(attributeName)).get(attributeName);
    }

    public Map<String, Optional<YTreeNode>> tryReadTableAttributes(YtTable table, Collection<String> attributeNames) {
        YTreeNode attrs = yt.cypress().get(table.ypath(), attributeNames);
        return listToMap(attributeNames, Function.identity(),
                attributeName -> attrs.getAttribute(attributeName));
    }

    public List<YTreeMapNode> getSchema(YtTable table) {
        YTreeNode node = yt.cypress().get(table.ypath(), Cf.wrap(Collections.singleton(YtUtils.SCHEMA_ATTR)));
        YTreeListNode schema = node.entityNode().getAttribute(YtUtils.SCHEMA_ATTR).get().listNode();

        List<YTreeMapNode> result = new ArrayList<>();
        schema.forEach(i -> result.add(i.mapNode()));
        return result;
    }

    public Map<String, YTreeNode> getAttributes(YtTable table, Set<String> attributeNames) {
        YTreeMapNode attrs = yt.cypress().get(table.ypath(), Cf.wrap(attributeNames)).mapNode();

        return attributeNames.stream()
                .collect(Collectors.toMap(Function.identity(), a -> attrs.get(a).orElse(null)));
    }

    public List<YTreeMapNode> selectRows(String query) {
        List<YTreeMapNode> rows = new ArrayList<>();
        yt.tables().selectRows(
                query,
                Optional.empty(),
                Optional.empty(),
                Optional.empty(),
                true,
                YTableEntryTypes.YSON,
                (Consumer<YTreeMapNode>) rows::add);
        return rows;
    }

    public List<YTreeMapNode> selectRows(String query, int inputRowLimit, int outputRowLimit) {
        List<YTreeMapNode> rows = new ArrayList<>();
        yt.tables().selectRows(
                query,
                Optional.empty(),
                Optional.of(inputRowLimit),
                Optional.of(outputRowLimit),
                true,
                YTableEntryTypes.YSON,
                (Consumer<YTreeMapNode>) rows::add);
        return rows;
    }

    /**
     * Считать таблицу из YT целиком, передав каждый ее ряд в consumer
     */
    public <T extends YtTableRow> void readTable(YtTable table, T tableRow, Consumer<T> consumer) {
        yt.tables().read(
                table.ypath(tableRow.getFields()),
                YTableEntryTypes.YSON,
                new ConvertConsumer<>(consumer, tableRow)
        );
    }

    /**
     * Считать часть записей таблицы из YT ограниченную заданными ключами, передав каждый считанный ряд в consumer
     *
     * @param table    таблица
     * @param consumer исполняемый код для считывания
     * @param tableRow объект описывающий ряд таблицы
     * @param start    первичный ключ, с которого начинам считывать (включительно)
     * @param end      первичный ключ, с которого заканчиваем считывать (исключая)
     */
    public <T extends YtTableRow> void readTableByKeyRange(YtTable table, Consumer<T> consumer, T tableRow, long start,
                                                           long end) {
        //Стоит обратить внимание, что в этом методе ключи в запросе будут считаться int'ами и long может не влезть.
        readTableByKeyRange(table, consumer, tableRow, YTree.integerNode(start), YTree.integerNode(end));
    }

    /**
     * Считать часть записей таблицы из YT ограниченную заданными ключами, передав каждый считанный ряд в consumer
     *
     * @param table    таблица
     * @param consumer исполняемый код для считывания
     * @param tableRow объект описывающий ряд таблицы
     * @param start    обертка над первичным ключом, с которого начинам считывать (включительно)
     * @param end      обертка над первичным ключом, с которого заканчиваем считывать (исключая)
     */
    public <T extends YtTableRow> void readTableByKeyRange(YtTable table, Consumer<T> consumer, T tableRow,
                                                           YTreeNode start,
                                                           YTreeNode end) {
        yt.tables().read(
                table.ypath().withRange(
                        new RangeLimit(Cf.list(start), -1, -1),
                        new RangeLimit(Cf.list(end), -1, -1)
                ),
                YTableEntryTypes.YSON,
                new ConvertConsumer<>(consumer, tableRow)
        );
    }

    /**
     * Считать часть записей таблицы из YT ограниченную заданными ключами, передав каждый считанный ряд в consumer
     *
     * @param table    таблица
     * @param consumer исполняемый код для считывания
     * @param tableRow объект описывающий ряд таблицы
     * @param start    номер строки, с которой начинам считывать (включительно)
     * @param end      номер строки, на которой заканчиваем считывать (исключая)
     */
    public <T extends YtTableRow> void readTableByRowRange(YtTable table, Consumer<T> consumer, T tableRow, long start,
                                                           long end) {
        yt.tables().read(
                table.ypath().withRange(start, end),
                YTableEntryTypes.YSON,
                new ConvertConsumer<>(consumer, tableRow)
        );
    }


    /**
     * Прочитать таблицу из YT целиком, лениво, с группировкой по {@code chunkSize} записей.
     * Метод защищен от модификации данных (удаления, модификации данных или полной перезаписи таблицы) при чтении,
     * так как делает это по id снепшота в рамках транзакции.
     * <p>
     * Потребитель данных ({@code consumer}) должен работать не дольше {@link #SNAPSHOT_TRANSACTION_TIMEOUT}.
     *
     * @param tablePath    путь к таблице
     * @param tableRowType тип строки в таблице
     * @param consumer     исполняемый код для считаывания групп объектов
     * @param chunkSize    сколько строк таблицы считывать за раз и передавать в {@code consumer}
     * @param <T>          тип строки таблицы
     * @return количество прочитанных строк
     * @implNote tableRow используется как буфер - это позволяет уменьшить нагрузку на GC
     */
    public <T> long readTableSnapshot(YPath tablePath, YTableEntryType<T> tableRowType, Consumer<List<T>> consumer,
                                      int chunkSize) {

        BiFunction<Transaction, YPath, Long> reader = (tx, path) -> {
            Optional<GUID> txId = Optional.of(tx.getId());

            AtomicLong rowsCount = new AtomicLong();
            try (var iterator = yt.tables()
                    .read(txId, true, path, tableRowType)
            ) {
                var subIterator = Cf.wrap(iterator)
                        .map(node -> {
                            rowsCount.incrementAndGet();
                            return node;
                        });
                Iterables.partition(() -> subIterator, chunkSize).forEach(consumer);
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new ReadException(e);
            } catch (Exception e) {
                throw new ReadException(e);
            }

            return rowsCount.longValue();
        };

        return YtTransactionsUtils.withSnapshotLock(yt,
                SNAPSHOT_TRANSACTION_TIMEOUT,
                Optional.of(SNAPSHOT_TRANSACTION_PING_PERIOD),
                tablePath,
                reader
        );
    }

    /**
     * Аналог метода ниже, берёт от таблицы те колонки, что описаны в tableRow
     *
     * @param table        таблица
     * @param tableRow     объект описывающий строку таблицы
     * @param rowConverter метод, превращающий ряд таблицы в объект (он не должен ссылаться на поля ряда!)
     * @param consumer     исполняемый код для считаывания групп объектов
     * @param chunkSize    сколько строк таблицы считывать за раз и передавать в {@code consumer}
     * @param <T>          тип строки таблицы
     * @param <R>          тип бизнес-объекта, передаваемого в {@code consumer}
     * @return количество прочитанных строк
     * @implNote tableRow используется как буфер - это позволяет уменьшить нагрузку на GC
     */
    public <T extends YtTableRow, R> long readTableSnapshot(YtTable table, T tableRow, Function<T, R> rowConverter,
                                                            Consumer<List<R>> consumer, int chunkSize) {
        return readTableSnapshot(table.ypath(tableRow.getFields()), tableRow, rowConverter, consumer, chunkSize);
    }

    /**
     * Прочитать таблицу из YT целиком, лениво, с группировкой по {@code chunkSize} записей.
     * Метод защищен от модификации данных (удаления, модификации данных или полной перезаписи таблицы) при чтении,
     * так как делает это по id снепшота в рамках транзакции.
     * <p>
     * Потребитель данных ({@code consumer}) должен работать не дольше {@link #SNAPSHOT_TRANSACTION_TIMEOUT}.
     *
     * @param tableYPath   объект, описывающий таблицу с опциональными параметрами вроде диапазона и набора колонок
     * @param tableRow     объект, описывающий строку таблицы
     * @param rowConverter метод, превращающий ряд таблицы в объект (он не должен ссылаться на поля ряда!)
     * @param consumer     исполняемый код для считаывания групп объектов
     * @param chunkSize    сколько строк таблицы считывать за раз и передавать в {@code consumer}
     * @param <T>          тип строки таблицы
     * @param <R>          тип бизнес-объекта, передаваемого в {@code consumer}
     * @return количество прочитанных строк
     */
    public <T extends YtTableRow, R> long readTableSnapshot(YPath tableYPath, T tableRow, Function<T, R> rowConverter,
                                                            Consumer<List<R>> consumer, int chunkSize) {

        BiFunction<Transaction, YPath, Long> reader = (tx, path) -> {
            Optional<GUID> txId = Optional.of(tx.getId());

            AtomicLong rowsCount = new AtomicLong();
            try (var iterator = yt.tables()
                    .read(txId, true, path, YTableEntryTypes.YSON)
            ) {
                var subIterator = Cf.wrap(iterator)
                        .map(node -> {
                            rowsCount.incrementAndGet();
                            tableRow.setData(node);
                            return rowConverter.apply(tableRow);
                        });
                Iterables.partition(() -> subIterator, chunkSize).forEach(consumer);
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new ReadException(e);
            } catch (Exception e) {
                throw new ReadException(e);
            }

            return rowsCount.longValue();
        };

        return YtTransactionsUtils.withSnapshotLock(yt,
                SNAPSHOT_TRANSACTION_TIMEOUT,
                Optional.of(SNAPSHOT_TRANSACTION_PING_PERIOD),
                tableYPath,
                reader
        );
    }

    /**
     * Произвести читающий запрос к динамической таблице, передав каждый ряд полученного результата в consumer
     *
     * @param table    таблица, к которой делается запрос
     * @param rowSpec  инстанс класса, описывающего ряд полученный из таблицы
     * @param where    строка с условием выбора данных
     * @param consumer класс обрабатывающий каждый полученный ряд
     * @param <T>      тип класса, описывающего ряд полученный из таблицы
     */
    public <T extends YtTableRow> void selectRows(YtTable table, T rowSpec, String where, Consumer<T> consumer) {
        String fields = String.join(", ", rowSpec.getFields().stream()
                .map(YtField::getName)
                .collect(Collectors.toList()));
        String query = String.format("%s FROM [%s] WHERE %s", fields, table.getPath(), where);
        yt.tables().selectRows(
                query,
                YTableEntryTypes.YSON,
                new ConvertConsumer<>(consumer, rowSpec)
        );
    }

    /**
     * Примонтировать динамическую таблицу
     */
    public void mountTable(YtTable table) {
        yt.tables().mount(table.ypath());
    }

    /**
     * Сделать статическую таблицу динамической
     * (таблица должна удовлетворять условиям из https://wiki.yandex-team
     * .ru/yt/userdoc/dynamictablesmapreduce/#prevrashheniestaticheskojjtablicyvdinamicheskuju)
     */
    public void makeTableDynamic(YtTable table) {
        yt.tables().alterTable(table.ypath(), Optional.of(true), Optional.empty());
    }

    /**
     * Отмонтировать динамическую таблицу
     */
    public void unMountTable(YtTable table) {
        yt.tables().unmount(table.ypath());
    }

    /**
     * Удалить таблицу
     */
    public void removeTable(YtTable table) {
        yt.cypress().remove(table.ypath());
    }

    /**
     * Считать таблицу из YT и вернуть список с ее содержимым
     */
    public <T extends YtTableRow, R> List<R> readTableAndMap(YtTable table, T tableRow, Function<T, R> mapper) {
        List<R> result = new ArrayList<>();
        yt.tables().read(
                table.ypath(tableRow.getFields()),
                YTableEntryTypes.YSON,
                new ConvertConsumer<>(r -> result.add(mapper.apply(r)), tableRow)
        );
        return result;
    }

    /**
     * Считать колонки таблицы из YT и вернуть список со строками
     */
    public List<YTreeMapNode> readTable(YtTable table, Collection<YtField> fields) {
        List<YTreeMapNode> result = new ArrayList<>();
        yt.tables().read(table.ypath(fields), YTableEntryTypes.YSON, (Consumer<YTreeMapNode>) result::add);
        return result;
    }

    /**
     * Считать одно поле из таблицы в  YT и вернуть список со всеми значениями
     */
    public <R> List<R> readTableField(YtTable table, YtField<R> field) {
        List<R> result = new ArrayList<>();
        yt.tables().read(
                table.ypath(Collections.singleton(field)),
                YTableEntryTypes.YSON,
                (Consumer<YTreeMapNode>) n -> result.add(field.extractValue(n))
        );
        return result;
    }

    public boolean exists(YtTable table) {
        return yt.cypress().exists(table.ypath());
    }

    private static class ConvertConsumer<T extends YtTableRow> implements Consumer<YTreeMapNode> {
        private final Consumer<T> baseConsumer;
        private final T tableRow;

        private ConvertConsumer(Consumer<T> baseConsumer, T tableRow) {
            this.baseConsumer = baseConsumer;
            this.tableRow = tableRow;
        }

        @Override
        public void accept(YTreeMapNode node) {
            tableRow.setData(node);
            baseConsumer.accept(tableRow);
        }
    }

    /**
     * Выполнить YQL-запрос
     */
    public void yqlExecute(String sql, Object... args) {
        yqlExecute(new YqlQuery(sql, args));
    }

    public void yqlExecute(YqlQuery query) {
        try (
                var ignored = Trace.current().profile("yql:execute", cluster.getName());
                var conn = yqlDataSource().getConnection();
                var st = createYqlPreparedStatement(conn, query)
        ) {
            var resultSetFuture = (ResultSetFuture) st.beginExecuteQuery();
            logger.info("Yql query started: {}", yqlOperationUrl(resultSetFuture));

            resultSetFuture.get();
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new RuntimeYqlException(e);
        } catch (SQLException | ExecutionException e) {
            throw new RuntimeYqlException(e);
        }
    }

    /**
     * Выполнить YQL-запрос
     */
    public <T> List<T> yqlQuery(String sql, YqlRowMapper<T> mapper, Object... args) {
        return yqlQuery(new YqlQuery(sql, args), mapper);
    }

    public <T> List<T> yqlQuery(YqlQuery query, YqlRowMapper<T> mapper) {
        try (
                var ignored = Trace.current().profile("yql:execute", cluster.getName());
                var conn = yqlDataSource().getConnection();
                var st = createYqlPreparedStatement(conn, query);
        ) {
            var resultSetFuture = (ResultSetFuture) st.beginExecuteQuery();
            logger.info("Yql query started: {}", yqlOperationUrl(resultSetFuture));

            try (var rs = resultSetFuture.get()) {
                List<T> result = new ArrayList<>();
                while (rs.next()) {
                    result.add(mapper.mapRowSafely(rs));
                }
                return result;
            }
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new RuntimeYqlException(e);
        } catch (SQLException | ExecutionException e) {
            throw new RuntimeYqlException(e);
        }
    }

    public ResultSetFuture yqlQueryBegin(YqlQuery query) {
        try (
                var ignored = Trace.current().profile("yql:begin_execute", cluster.getName());
                var conn = yqlDataSource().getConnection();
                var st = createYqlPreparedStatement(conn, query);
        ) {
            var resultSetFuture = (ResultSetFuture) st.beginExecuteQuery();
            logger.info("Yql query started: {}", yqlOperationUrl(resultSetFuture));
            return resultSetFuture;
        } catch (SQLException e) {
            throw new RuntimeYqlException(e);
        }
    }

    private YqlPreparedStatement createYqlPreparedStatement(Connection connection, YqlQuery query) throws SQLException {
        var tunedSql = tuneYql(query.getSql());
        YqlPreparedStatement st = (YqlPreparedStatement) connection.prepareStatement(tunedSql);

        if (query.getTitle() != null) {
            st.setTitle("YQL: " + query.getTitle());
        } else {
            st.setTitle("YQL: " + Trace.current().getMethod());
        }
        if (query.getAcl() != null) {
            st.setAcl(query.getAcl());
        } else {
            var defaultAcl = ytClusterConfig.getDefaultYqlAcl();
            if (defaultAcl != null) {
                st.setAcl(defaultAcl);
            }
        }

        var args = query.getBinds();
        for (int i = 0; i < args.length; i++) {
            st.setObject(i + 1, args[i]);
        }

        return st;
    }

    /**
     * Добавляем некоторые пргмы, которые мы хотим видеть в запросах почти всегда
     */
    private String tuneYql(String sql) {
        var prefix = new StringBuilder();
        if (!sql.contains("yt.TemporaryAutoMerge")) {
            prefix.append("PRAGMA yt.TemporaryAutoMerge='disabled';\n");
        }
        return prefix.length() == 0
                ? sql
                : "\n-- auto header begin\n" + prefix.toString() + "-- auto header end\n" + sql;
    }

    private String yqlOperationUrl(ResultSetFuture fut) {
        return YQL_OPERATION_URL_BASE + fut.getOperationId();
    }

    private YqlDataSource yqlDataSource() {
        return ytProvider.getYql(cluster, ytSQLSyntaxVersion);
    }

    @Override
    public String toString() {
        return String.format("YtOperator(cluster=%s)", cluster);
    }

    /**
     * Возвращает true, если таблеты динтаблицы находятся в состоянии "frozen".
     * Иначе возвращает false.
     */
    public boolean isFrozen(YPath table) {
        return "frozen".equals(getTabletState(table));
    }

    /**
     * Инициирует заморозку указанной таблицы и опрашивает с экспоненциально возрастающим
     * временем ожидания статус таблетов (изначально 100 мс).
     * Если таблица не успевает перейти в состояние "frozen" за timeoutMs миллисекунд, будет выброшено исключение.
     */
    public void freeze(YPath table, long timeoutMs) {
        Stopwatch sw = Stopwatch.createStarted();
        getYt().tables().freeze(table);

        int intervalMs = 100;
        while (!isFrozen(table)) {
            if (sw.elapsed(TimeUnit.MILLISECONDS) >= timeoutMs) {
                throw new RuntimeException("Can't freeze a table " + table);
            }
            try {
                Thread.sleep(intervalMs);
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new RuntimeException("Thread was interrupted", e);
            }
            intervalMs += intervalMs / 10;
        }
    }

    /**
     * Инициирует разморозку указанной таблицы и опрашивает с экспоненциально возрастающим
     * временем ожидания статус таблетов (изначально 100 мс).
     * Если таблица не успевает перейти в состояние "mounted" за timeoutMs миллисекунд, будет выброшено исключение.
     */
    public void unfreeze(YPath table, long timeoutMs) {
        Stopwatch sw = Stopwatch.createStarted();
        getYt().tables().unfreeze(table);

        int intervalMs = 100;
        while (!isMounted(table)) {
            if (sw.elapsed(TimeUnit.MILLISECONDS) >= timeoutMs) {
                throw new RuntimeException("Can't unfreeze a table " + table);
            }
            try {
                Thread.sleep(intervalMs);
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new RuntimeException("Thread was interrupted", e);
            }
            intervalMs += intervalMs / 10;
        }
    }

    /**
     * Возвращает true, если таблеты динтаблицы находятся в состоянии "mounted".
     * Иначе возвращает false.
     */
    public boolean isMounted(YPath table) {
        return "mounted".equals(getTabletState(table));
    }

    /**
     * Возвращает true, если таблеты динтаблицы находятся в состоянии "unmounted".
     * Иначе возвращает false.
     */
    public boolean isUnmounted(YPath table) {
        return "unmounted".equals(getTabletState(table));
    }

    public String getTabletState(YPath table) {
        YTreeNode tabletState = getYt().cypress().get(table, new SingletonSet<>("tablet_state"));
        return tabletState.getAttribute("tablet_state").get().stringValue();
    }

    /**
     * Инициирует монтирование указанной таблицы и опрашивает с экспоненциально возрастающим
     * временем ожидания статус таблетов (изначально 100 мс).
     * Если таблица не успевает перейти в состояние "mounted" за timeoutMs миллисекунд, будет выброшено исключение.
     */
    public void mount(YPath table, int timeoutMs) {
        Stopwatch sw = Stopwatch.createStarted();
        getYt().tables().mount(table);

        int intervalMs = 100;
        while (!isMounted(table)) {
            if (sw.elapsed(TimeUnit.MILLISECONDS) >= timeoutMs) {
                throw new RuntimeException("Can't mount a table " + table);
            }
            try {
                Thread.sleep(intervalMs);
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new RuntimeException("Thread was interrupted", e);
            }
            intervalMs += intervalMs / 10;
        }
    }

    /**
     * Инициирует размонтирование указанной таблицы и опрашивает с экспоненциально возрастающим
     * временем ожидания статус таблетов (изначально 100 мс).
     * Если таблица не успевает перейти в состояние "unmounted" за timeoutMs миллисекунд, будет выброшено исключение.
     */
    public void unmount(YPath table, int timeoutMs) {
        Stopwatch sw = Stopwatch.createStarted();
        getYt().tables().unmount(table);

        int intervalMs = 100;
        while (!isUnmounted(table)) {
            if (sw.elapsed(TimeUnit.MILLISECONDS) >= timeoutMs) {
                throw new RuntimeException("Can't unmount table " + table);
            }
            try {
                Thread.sleep(intervalMs);
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new RuntimeException("Thread was interrupted", e);
            }
            intervalMs += intervalMs / 10;
        }
    }
}
